import React, {useCallback} from 'react';

import {URLs} from 'constants/urls';
import {EAppActions} from 'constants/platforms/TPlatforms';

import {IWithDeviceType} from 'types/withDeviceType';

import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';
import {deviceMods} from 'utilities/stylesUtils';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {internalUrl} from 'utilities/url';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUserInfo} from 'utilities/hooks/useUserInfo';

import * as i18nFavoritesBlock from 'i18n/account-Favorites';

import TicketCatL from 'icons/illustrations/TicketCat/TicketCatL';
import TicketCatM from 'icons/illustrations/TicketCat/TicketCatM';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import HeartOutlineIcon from 'icons/16/HeartOutline';
import useReturnPathUrl from 'components/User/hooks/useReturnPathUrl';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './EmptyFavorites.scss';

export interface IEmptyFavoritesProps extends IWithDeviceType {}

const EmptyFavorites: React.FC<IEmptyFavoritesProps> = ({deviceType}) => {
    const userInfo = useUserInfo();
    const isAuth = isAuthUser(userInfo);
    const returnPathUrl = useReturnPathUrl();
    const coordinator = useCoordinator();

    const handleFindHotelsButtonClick: React.MouseEventHandler = useCallback(
        e => {
            /*
             * мы на фронте отправляем на страницу поиска отелей
             * в приложении идёт переход на морду
             */
            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_INDEX_PAGE,
            );

            if (processed) {
                e.preventDefault();
            }
        },
        [coordinator],
    );

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            {deviceType.isDesktop ? (
                <TicketCatL width={210} height={210} />
            ) : (
                <TicketCatM width={120} height={120} />
            )}
            <span className={cx('title')}>
                {i18nFavoritesBlock.noFavoriteHotels()}
            </span>
            <span
                className={cx('description', {description_inOneLine: !isAuth})}
            >
                {isAuth
                    ? insertJSXIntoKey(
                          i18nFavoritesBlock.noFavoriteHotelsDescription,
                      )({
                          heartIcon: (
                              <HeartOutlineIcon
                                  height={14}
                                  className={cx('icon')}
                              />
                          ),
                      })
                    : i18nFavoritesBlock.noFavoriteHotelsDescriptionNoLogin()}
            </span>
            {isAuth ? (
                <ButtonLink
                    to={internalUrl(URLs.hotelsSearch)}
                    theme="primary"
                    className={cx('findHotelsButton')}
                    onClick={handleFindHotelsButtonClick}
                >
                    {i18nFavoritesBlock.findHotels()}
                </ButtonLink>
            ) : (
                <ButtonLink
                    url={getPassportUrl({
                        mode: EPassportMode.AUTH,
                        passportHost: userInfo.passportPath,
                        retpath: returnPathUrl,
                    })}
                    theme="primary"
                    className={cx('findHotelsButton')}
                >
                    {i18nFavoritesBlock.signIn()}
                </ButtonLink>
            )}
        </div>
    );
};

export default EmptyFavorites;
