import React, {useCallback, memo} from 'react';
import {useSelector, useDispatch} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {changeFavoriteCategoryAction} from 'reducers/favorites/actions';

import {getFavoritesInfoSelector} from 'selectors/favorites/getFavoritesInfo';

import {deviceMods} from 'utilities/stylesUtils';

import DesktopFavoriteCategories from './DesktopFavoriteCategories/DesktopFavoriteCategories';
import MobileFavoriteCategories from './MobileFavoriteCategories/MobileFavoriteCategories';

import cx from './FavoriteCategories.scss';

export interface ICategory {
    key: string;
    value: string | React.ReactElement;
}

export interface IFavoriteCategoriesProps
    extends IWithClassName,
        IWithDeviceType {
    onClick?: () => void;
}

const FavoriteCategories: React.FC<IFavoriteCategoriesProps> = ({
    className,
    deviceType,
    onClick,
}) => {
    const {categories, selectedCategoryId} = useSelector(
        getFavoritesInfoSelector,
    );
    const dispatch = useDispatch();
    const onCategoryChange = useCallback(
        (category: ICategory) => {
            dispatch(changeFavoriteCategoryAction(category.key));
            onClick?.();
        },
        [dispatch, onClick],
    );

    return (
        <div className={cx('root', deviceMods('root', deviceType), className)}>
            {deviceType.isDesktop ? (
                <DesktopFavoriteCategories
                    onCategoryChange={onCategoryChange}
                    categories={categories || []}
                    selectedCategoryId={selectedCategoryId || ''}
                />
            ) : (
                <MobileFavoriteCategories
                    onCategoryChange={onCategoryChange}
                    categories={categories || []}
                    selectedCategoryId={selectedCategoryId || ''}
                />
            )}
        </div>
    );
};

export default memo(FavoriteCategories);
