import React, {useCallback, memo} from 'react';
import {useSelector, useDispatch} from 'react-redux';

import {FAVORITE_HOTELS_SEARCH_FORM_NAME} from 'constants/reduxForm';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    ESearchFormSize,
    ESearchFormTriggerViewType,
    ESearchFormFieldName,
} from 'components/SearchForm/types';
import {ISearchFormRenderTravelersBlockProps} from 'components/SearchForm/types/ISearchFormRenderTravelersBlockProps';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';
import {ECalendarType} from 'components/Calendar/types';

import {
    setCheckinDateAction,
    setCheckoutDateAction,
    setAdultsAction,
    setChildrenAgesAction,
} from 'reducers/favorites/actions';
import {previousSearchFormSetFieldValue} from 'reducers/common/previousSearchForm/actions';

import {getFavoritesInfoSelector} from 'selectors/favorites/getFavoritesInfo';

import {deviceMods} from 'utilities/stylesUtils';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import validateForm from 'projects/hotels/components/SearchForm/utilities/validateForm';
import getSuggestTitleAndDescription from 'projects/hotels/components/SearchForm/utilities/getSuggestTitleAndDescription';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/components';
import * as favoritesKeyset from 'i18n/account-Favorites';

import CommonSearchForm, {
    IOnDateClickParams,
} from 'components/SearchForm/SearchForm';
import HotelsTravellers from 'projects/hotels/components/SearchForm/components/HotelsTravellers/HotelsTravellers';
import {useNewCalendarEnabled} from 'components/DatePicker/hooks/useNewCalendarEnabled';

const FIELD_NAMES = [
    ESearchFormFieldName.START_DATE,
    ESearchFormFieldName.END_DATE,
];

import cx from './FavoriteHotelsSearchForm.scss';

export interface ICategory {
    key: string;
    value: string | React.ReactElement;
}

export interface IFavoriteHotelsSearchFormProps
    extends IWithClassName,
        IWithDeviceType {
    onSubmit: () => void;
}

const FavoriteHotelsSearchForm: React.FC<IFavoriteHotelsSearchFormProps> = ({
    className,
    deviceType,
    onSubmit,
}) => {
    const {offerSearchParams} = useSelector(getFavoritesInfoSelector);
    const {checkinDate, checkoutDate, adults, childrenAges} =
        offerSearchParams || {};
    const dispatch = useDispatch();

    const setStartDate = useDispatchedAction(setCheckinDateAction, dispatch);
    const setEndDate = useDispatchedAction(setCheckoutDateAction, dispatch);
    const setAdults = useDispatchedAction(setAdultsAction, dispatch);
    const setChildrenAges = useDispatchedAction(
        setChildrenAgesAction,
        dispatch,
    );

    const handleFormSubmit = useCallback(() => {
        onSubmit();
        reachGoal(EFavoritesGoal.FIND_CLICK);
    }, [onSubmit]);

    const isNewCalendarEnabled = useNewCalendarEnabled();

    const onDateClick = useCallback(
        (args?: IOnDateClickParams) => {
            if (args) {
                const {startDate, endDate, calendarType} = args;

                /** TODO: Есть отдельный таск на интеграцию метрик, логика в нем немного изменена
                 * после окончания задачи, нужно будет подтянуть сюда метрики
                 * @see https://st.yandex-team.ru/HOTELS-5619 */
                if (isNewCalendarEnabled) {
                    return;
                }

                if (
                    startDate &&
                    endDate &&
                    Number(startDate) === Number(endDate)
                ) {
                    if (calendarType === ECalendarType.StartDate) {
                        setEndDate('');
                    } else {
                        setStartDate('');
                    }
                }

                if (calendarType === ECalendarType.StartDate && startDate) {
                    reachGoal(EFavoritesGoal.CHECKIN_DATE);
                }

                if (calendarType === ECalendarType.EndDate && endDate) {
                    reachGoal(EFavoritesGoal.CHECKOUT_DATE);
                }
            }
        },
        [setStartDate, setEndDate],
    );

    const setPreviousSearchFormValue = useDispatchedAction(
        previousSearchFormSetFieldValue,
        dispatch,
    );

    const renderTravellersBlock = useCallback(
        ({baseCx}: ISearchFormRenderTravelersBlockProps) => {
            return (
                <div className={cx(baseCx('travellers'), cx('travellers'))}>
                    <HotelsTravellers
                        triggerClassName={baseCx('travellersTrigger')}
                        triggerFocusClassName={baseCx(
                            'travellersTrigger_focus',
                        )}
                        size={
                            deviceType.isMobile
                                ? ESearchFormSize.M
                                : ESearchFormSize.S
                        }
                        triggerViewType={ESearchFormTriggerViewType.TILE}
                        adultsCount={adults || 1}
                        setAdults={setAdults}
                        childrenAges={childrenAges || []}
                        setChildrenAges={setChildrenAges}
                        deviceType={deviceType}
                        canToggle={deviceType.isDesktop}
                        onBlur={(): void => {}}
                    />
                </div>
            );
        },
        [adults, childrenAges, deviceType, setAdults, setChildrenAges],
    );

    return (
        <div className={cx('root', deviceMods('root', deviceType), className)}>
            <CommonSearchForm
                className={cx('searchForm')}
                submitButtonClassName={cx('submitButton')}
                size={
                    deviceType.isMobile ? ESearchFormSize.M : ESearchFormSize.S
                }
                triggerViewType={ESearchFormTriggerViewType.TILE}
                searchFormName={FAVORITE_HOTELS_SEARCH_FORM_NAME}
                fieldsNames={FIELD_NAMES}
                startDateField={checkinDate || null}
                endDateField={checkoutDate || null}
                hasEndDate
                toSearchSuggests={null}
                validateForm={validateForm}
                uniqueSuggestValueName="id"
                getSuggestTitleAndDescription={getSuggestTitleAndDescription}
                toPointPlaceholder=""
                startDatePlaceholder={i18nBlock.hotelsSearchFormDotStartDatePlaceholder()}
                endDatePlaceholder={i18nBlock.hotelsSearchFormDotEndDatePlaceholder()}
                setStartDate={setStartDate}
                setEndDate={setEndDate}
                setPreviousSearchFormValue={setPreviousSearchFormValue}
                onSubmit={handleFormSubmit}
                onDateClick={onDateClick}
                isRoundTrip={true}
                renderTravellersBlock={renderTravellersBlock}
                submitButtonText={favoritesKeyset.show()}
                submitButtonTheme="primary"
                {...prepareQaAttributes('search-form')}
            />
        </div>
    );
};

export default memo(FavoriteHotelsSearchForm);
