import React, {useCallback, useRef, useState} from 'react';
import {useDispatch} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import IShareFavoriteHotelsRequestParams from 'server/api/HotelsFavoritesAPI/types/IShareFavoriteHotelsRequestParams';

import {setSnackbarInfo} from 'reducers/common/snackbar/actions';
import {ISnackbarState} from 'reducers/common/snackbar/reducer';
import {setShareTokenAction} from 'reducers/favorites/actions';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    EShareStatus,
    useNativeShareAPI,
} from 'utilities/hooks/useNativeShareAPI';
import {logWarning} from 'utilities/logger/logWarning';
import getShareUrl from 'projects/favorites/utilities/getShareUrl';
import {getAttributionParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';

import * as i18nBlock from 'i18n/common-share';

import SharePopup from 'components/Share/compoenents/SharePopup/SharePopup';

import {hotelsFavoritesService} from 'serviceProvider';

export interface IShareProps extends IWithClassName {
    button: React.ReactElement;
    shareTokenParams: IShareFavoriteHotelsRequestParams;

    onClick?: () => void;
    onCopyLinkClick?: () => void;

    page: ISnackbarState['page'];
}

const Share: React.FC<IShareProps> = ({
    className,
    button,
    shareTokenParams,
    onClick,
    onCopyLinkClick,
    page,
}) => {
    const {
        value: popupIsOpen,
        toggle: changePopupIsOpened,
        setFalse: closePopup,
    } = useBoolean(false);
    const [link, setLink] = useState<string>();
    const dispatch = useDispatch();
    const buttonRef = useRef<HTMLDivElement>(null);

    const shareViaNativeAPI = useNativeShareAPI();

    const handleClick = useCallback(async () => {
        if (!popupIsOpen) {
            onClick?.();
        }

        let shareUrl;

        try {
            const {token} = await hotelsFavoritesService
                .provider()
                .shareFavoriteHotels({
                    ...shareTokenParams,
                    ...getAttributionParams(),
                });

            dispatch(setShareTokenAction(token));
            shareUrl = getShareUrl(token);
        } catch (err) {
            logWarning(
                {
                    message:
                        '[YATRAVEL] Не удалось получить токен для шаринга избранного',
                },
                err,
            );
        }

        if (shareUrl) {
            const nativeStatus = await shareViaNativeAPI({url: shareUrl});

            setLink(shareUrl);

            if (nativeStatus.status === EShareStatus.OK) {
                onCopyLinkClick?.();

                return;
            }

            if (nativeStatus.status !== EShareStatus.UNSUPPORTED) {
                return;
            }

            changePopupIsOpened();
        }
    }, [
        popupIsOpen,
        shareTokenParams,
        shareViaNativeAPI,
        changePopupIsOpened,
        onClick,
        onCopyLinkClick,
    ]);

    const handleCopyLink = useCallback(() => {
        onCopyLinkClick?.();

        dispatch(
            setSnackbarInfo({
                lastAction: `copy-${link}`,
                hasCancellation: false,
                message: i18nBlock.linkCopied(),
                page,
            }),
        );
    }, [dispatch, link, onCopyLinkClick, page]);

    return (
        <div className={className}>
            <div onClick={handleClick} ref={buttonRef}>
                {button}
            </div>

            {link && (
                <SharePopup
                    isVisible={popupIsOpen}
                    onClose={closePopup}
                    link={link}
                    onCopyLinkClick={handleCopyLink}
                    popupAnchor={buttonRef}
                />
            )}
        </div>
    );
};

export default Share;
