import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {
    ALL_CATEGORY_NAME,
    MAX_FAVORITE_HOTELS_ON_PAGE,
} from 'projects/favorites/constants';
import {EProjectName} from 'constants/common';

import {INavigationTokens} from 'types/hotels/search/INavigationTokens';
import {EFooterProject} from 'components/Footer/types';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';

import {
    changeFavoriteCategoryAction,
    getFavoritesInfoActions,
    removeFavoriteHotelsAction,
    resetActiveHotel as resetActiveHotelAction,
    setActiveHotel as setActiveHotelAction,
} from 'reducers/favorites/actions';

import {getFavoritesInfoSelector} from 'selectors/favorites/getFavoritesInfo';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {reachGoal} from 'utilities/metrika';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import useQuery from 'utilities/hooks/useQuery';
import updateLocationWithQuery from 'utilities/updateLocationWithQuery/updateLocationWithQuery';
import useAddFavoriteHotelClick from 'projects/favorites/utilities/useAddFavoriteHotelClick';
import {useIsAuth} from 'utilities/hooks/useIsAuth';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as favoritesKeyset from 'i18n/account-Favorites';
import * as commonKeyset from 'i18n/common';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Spinner from 'components/Spinner/Spinner';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import LayoutDefault, {
    ELayoutBgColor,
    ILayoutDefaultProps,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import HotelsList, {
    EHotelsListSkeletonType,
} from 'projects/hotels/components/HotelsList/HotelsList';
import FavoriteCategories from 'projects/favorites/components/FavoriteCategories/FavoriteCategories';
import NavigationButtons from 'projects/hotels/components/NavigationButtons/NavigationButtons';
import FavoriteHotelsSearchForm from 'projects/favorites/components/FavoriteHotelsSearchForm/FavoriteHotelsSearchForm';
import DeleteFavoriteCategoryWarningModal from 'projects/favorites/components/DeleteFavoriteCategoryWarningModal/DeleteFavoriteCategoryWarningModal';
import EmptyFavorites from 'projects/favorites/components/EmptyFavorites/EmptyFavorites';
import DeleteIcon from 'icons/16/Delete';
import ShareFieldIcon from 'icons/16/ShareFilled';
import Back from 'components/ModalWithBackButton/components/Back/Back';
import HotelSearchCardSkeleton from 'projects/hotels/pages/SearchPage/components/MobileHotelSearchCardSkeleton/MobileHotelSearchCardSkeleton';
import InfiniteScrollTrigger from 'components/InfiniteScrollTrigger/InfiniteScrollTrigger';
import ForceLoginAuthLink from 'components/ForceLogin/ForceLoginAuthLink/ForceLoginAuthLink';
import ShareFavorites from 'projects/favorites/components/ShareFavorites/ShareFavorites';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './FavoritesPage.scss';

const NAVIGATION_TOKENS: INavigationTokens = {
    nextPage: 'nextPage',
    prevPage: 'prevPage',
    currentPage: 'currentPage',
};

function getOffset(token: string, offset: number): number {
    if (token === NAVIGATION_TOKENS.nextPage) {
        return offset + MAX_FAVORITE_HOTELS_ON_PAGE;
    }

    if (token === NAVIGATION_TOKENS.prevPage) {
        return offset - MAX_FAVORITE_HOTELS_ON_PAGE;
    }

    return offset;
}

export interface IFavoritesPageProps {}

const FavoritesPage: React.FC<IFavoritesPageProps> = () => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const {isWeb} = usePlatform();
    const {categoryId} = useQuery(['categoryId']);
    const {
        categories,
        selectedCategoryId,
        offerSearchParams,
        hotelWithOffersByPermalink,
        permalinksList,
        isError,
        isLoading,
        lastSearchTimestamp,
        offerSearchProgress,
        offset,
        totalHotelCount,
        activeHotelPermalink,
        nights,
    } = useSelector(getFavoritesInfoSelector);
    const isAuth = useIsAuth();

    const {
        setTrue: openDeleteFavoriteCategoryWarningModal,
        setFalse: closeDeleteFavoriteCategoryWarningModal,
        value: isOpenDeleteFavoriteCategoryWarningModal,
    } = useBoolean(false);

    const isCategoryView = Boolean(selectedCategoryId);
    const nextNavigationToken =
        totalHotelCount &&
        offset < totalHotelCount - MAX_FAVORITE_HOTELS_ON_PAGE
            ? NAVIGATION_TOKENS.nextPage
            : '';
    const hasHotels = useMemo(
        () =>
            Object.values(hotelWithOffersByPermalink || {}).filter(
                h => h.hotel.isFavorite,
            ).length > 0,
        [hotelWithOffersByPermalink],
    );

    const selectedCategory = useMemo(() => {
        const categoryToFind = selectedCategoryId || ALL_CATEGORY_NAME;

        return categories?.find(category => category.id === categoryToFind);
    }, [selectedCategoryId, categories]);

    useEffect(() => {
        if (selectedCategory && categories) {
            reachGoal(EFavoritesGoal.SELECTED_GROUP, {
                favorites: {
                    groupsCount: categories.length,
                    groupName: selectedCategory.name,
                    hotelsCount: selectedCategory.hotelCount,
                },
            });
        }
    }, [selectedCategory?.id]);

    useEffect(() => {
        if (categoryId) {
            dispatch(changeFavoriteCategoryAction(categoryId || null));
        }
    }, []);

    useEffect(() => {
        if (!categoryId && !selectedCategoryId) {
            dispatch(
                getFavoritesInfoActions.request({
                    offset,
                    limit: MAX_FAVORITE_HOTELS_ON_PAGE,
                }),
            );
        } else if (categoryId === selectedCategoryId) {
            dispatch(
                getFavoritesInfoActions.request({
                    offset,
                    limit: MAX_FAVORITE_HOTELS_ON_PAGE,
                    categoryId: categoryId || undefined,
                }),
            );
        } else {
            updateLocationWithQuery(
                {categoryId: selectedCategoryId},
                undefined,
                {
                    replace: false,
                    needSaveLocationState: true,
                    stringifyOptions: {skipNull: true},
                },
            );
        }
    }, [categoryId, selectedCategoryId]);

    const shareTokenParams = useMemo(
        () => ({
            adults: offerSearchParams?.adults || 2,
            childrenAges: offerSearchParams?.childrenAges || [],
            categoryId: selectedCategoryId || ALL_CATEGORY_NAME,
            checkinDate: offerSearchParams?.checkinDate || '',
            checkoutDate: offerSearchParams?.checkoutDate || '',
        }),
        [offerSearchParams, selectedCategoryId],
    );

    const resetCategory = useCallback(() => {
        dispatch(changeFavoriteCategoryAction(null));
    }, [dispatch]);

    const handleChangeNavigationToken = useCallback(
        (token: string) => {
            dispatch(
                getFavoritesInfoActions.request({
                    offset: getOffset(token, offset),
                    limit: MAX_FAVORITE_HOTELS_ON_PAGE,
                    categoryId: selectedCategoryId || undefined,
                    withInfiniteScroll: deviceType.isMobile,
                }),
            );
        },
        [dispatch, offset, selectedCategoryId, deviceType],
    );

    const handleLoadNextHotels = useCallback(() => {
        if (nextNavigationToken) {
            handleChangeNavigationToken(nextNavigationToken);
        }
    }, [handleChangeNavigationToken, nextNavigationToken]);

    const navigationButtons = useMemo(() => {
        const loading = !offerSearchProgress?.finished || isLoading;

        if (deviceType.isMobile) {
            return (
                <InfiniteScrollTrigger
                    isLoading={loading}
                    hasItems={Boolean(permalinksList?.length)}
                    onLoadNeeded={handleLoadNextHotels}
                    loadingNode={
                        <>
                            <HotelSearchCardSkeleton
                                className={cx('skeleton')}
                            />
                            <HotelSearchCardSkeleton
                                className={cx('skeleton')}
                            />
                        </>
                    }
                />
            );
        }

        return (
            <NavigationButtons
                className={cx(
                    'navigationButtons',
                    deviceMods('navigationButtons', deviceType),
                )}
                isLoading={loading}
                isEmptyList={!permalinksList?.length}
                navigationTokens={{
                    ...NAVIGATION_TOKENS,
                    prevPage: offset > 0 ? NAVIGATION_TOKENS.prevPage : '',
                    nextPage: nextNavigationToken,
                }}
                onChangeNavigationToken={handleChangeNavigationToken}
            />
        );
    }, [
        offerSearchProgress,
        permalinksList,
        isLoading,
        deviceType,
        offset,
        nextNavigationToken,
        handleChangeNavigationToken,
        handleLoadNextHotels,
    ]);

    const setActiveHotel = useDispatchedAction(setActiveHotelAction, dispatch);
    const resetActiveHotel = useDispatchedAction(
        resetActiveHotelAction,
        dispatch,
    ) as () => void;

    const deleteCategory = useCallback(() => {
        if (selectedCategoryId) {
            dispatch(
                removeFavoriteHotelsAction({categoryId: selectedCategoryId}),
            );
            dispatch(
                getFavoritesInfoActions.request({
                    offset: 0,
                    limit: MAX_FAVORITE_HOTELS_ON_PAGE,
                }),
            );
            closeDeleteFavoriteCategoryWarningModal();
        }
    }, [dispatch, selectedCategoryId, closeDeleteFavoriteCategoryWarningModal]);

    const handleShareClick = useReachGoal(EFavoritesGoal.SHARE_CLICK);
    const handleCopyLinkClick = useReachGoal(EFavoritesGoal.COPY_LINK_CLICK);

    const selectedCategoryName = useMemo(() => {
        if (selectedCategoryId === ALL_CATEGORY_NAME) {
            return favoritesKeyset.favorites();
        }

        return categories?.find(category => category.id === selectedCategoryId)
            ?.name;
    }, [selectedCategoryId, categories]);

    const handleSearchFormSubmit = useCallback(() => {
        dispatch(
            getFavoritesInfoActions.request({
                offset: 0,
                limit: MAX_FAVORITE_HOTELS_ON_PAGE,
                categoryId: selectedCategoryId || undefined,
            }),
        );
    }, [dispatch, selectedCategoryId]);

    const handleAddFavoriteHotelClick = useAddFavoriteHotelClick();

    const hotelsNode = useMemo(
        () => (
            <HotelsList
                className={cx(deviceMods('hotelsList', deviceType))}
                nights={nights || 0}
                deviceType={deviceType}
                searchInfoFromProps={
                    {
                        ...offerSearchParams,
                        isError,
                        isLoading,
                        lastSearchTimestamp,
                    } || undefined
                }
                hotels={permalinksList || undefined}
                hotelWithOffersByPermalink={
                    hotelWithOffersByPermalink || undefined
                }
                totalHotelsList={MAX_FAVORITE_HOTELS_ON_PAGE}
                onFavoriteClick={handleAddFavoriteHotelClick}
                searchIsFinished={offerSearchProgress?.finished}
                skeletonType={EHotelsListSkeletonType.OFFER_SKELETON}
                onHoverHotelCard={setActiveHotel}
                onLeaveHotelCard={resetActiveHotel}
                activeHotelPermalink={activeHotelPermalink}
                offerRequestParams={offerSearchParams || undefined}
                currentPage="favorites"
                hideNoFavoritesHotels
                snackbarClassName={cx('snackbar')}
            />
        ),
        [
            deviceType,
            nights,
            offerSearchParams,
            isError,
            isLoading,
            lastSearchTimestamp,
            permalinksList,
            hotelWithOffersByPermalink,
            handleAddFavoriteHotelClick,
            offerSearchProgress,
            setActiveHotel,
            resetActiveHotel,
            activeHotelPermalink,
        ],
    );

    const desktopContent = useMemo(() => {
        return isLoading ? (
            <Spinner size="xl" className={cx('spinner')} />
        ) : (
            <div className={cx('mainInfoContainer')}>
                <Flex justifyContent="space-between" alignItems="flex-end">
                    <Text size="xxl" weight="bold">
                        {selectedCategoryName}
                    </Text>

                    <Flex alignItems="flex-end">
                        {selectedCategoryId &&
                            selectedCategoryId !== ALL_CATEGORY_NAME && (
                                <TextWithIcon
                                    text={favoritesKeyset._delete()}
                                    iconLeft={DeleteIcon}
                                    className={cx(
                                        'textWithIcon',
                                        !hasHotels && 'hideButton',
                                    )}
                                    onClick={
                                        openDeleteFavoriteCategoryWarningModal
                                    }
                                />
                            )}

                        <ShareFavorites
                            button={
                                <TextWithIcon
                                    text={favoritesKeyset.share()}
                                    iconLeft={ShareFieldIcon}
                                    className={cx('textWithIcon')}
                                />
                            }
                            shareTokenParams={shareTokenParams}
                            onClick={handleShareClick}
                            className={cx(!hasHotels && 'hideButton')}
                            onCopyLinkClick={handleCopyLinkClick}
                            page="favorites"
                        />
                    </Flex>
                </Flex>

                <FavoriteHotelsSearchForm
                    deviceType={deviceType}
                    className={cx('favoriteHotelsSearchForm')}
                    onSubmit={handleSearchFormSubmit}
                />

                {!isAuth && (
                    <ForceLoginAuthLink
                        image="yandex"
                        className={cx('forceLogin')}
                        message={favoritesKeyset.forceLogin()}
                    />
                )}

                {hotelsNode}

                {navigationButtons}
            </div>
        );
    }, [
        selectedCategoryName,
        selectedCategoryId,
        openDeleteFavoriteCategoryWarningModal,
        deviceType,
        shareTokenParams,
        isLoading,
        handleShareClick,
        handleSearchFormSubmit,
        navigationButtons,
        hotelsNode,
        isAuth,
        hasHotels,
    ]);

    const greyBg = isCategoryView && deviceType.isMobile;

    const mobileContent = useMemo(() => {
        return (
            <Flex flexDirection="column">
                <div className={cx({headerBg: greyBg})}>
                    {isCategoryView && (
                        <Back
                            className={cx('backButton')}
                            onClick={resetCategory}
                            text={commonKeyset.back()}
                        />
                    )}
                    <Flex
                        justifyContent="space-between"
                        alignItems="center"
                        className={cx('header')}
                    >
                        <Text
                            size="xl"
                            weight="bold"
                            className={cx('categoryName')}
                        >
                            {selectedCategoryName}
                        </Text>

                        <Flex>
                            {isWeb && (
                                <ShareFavorites
                                    className={cx(!hasHotels && 'hideButton')}
                                    button={
                                        <ShareFieldIcon
                                            className={cx('icon')}
                                        />
                                    }
                                    shareTokenParams={shareTokenParams}
                                    onClick={handleShareClick}
                                    onCopyLinkClick={handleCopyLinkClick}
                                    page="favorites"
                                />
                            )}
                            {selectedCategoryId &&
                                selectedCategoryId !== ALL_CATEGORY_NAME && (
                                    <DeleteIcon
                                        onClick={
                                            openDeleteFavoriteCategoryWarningModal
                                        }
                                        className={cx(
                                            'icon',
                                            !hasHotels && 'hideButton',
                                        )}
                                    />
                                )}
                        </Flex>
                    </Flex>

                    <FavoriteHotelsSearchForm
                        deviceType={deviceType}
                        onSubmit={handleSearchFormSubmit}
                    />
                </div>

                {!isAuth && (
                    <ForceLoginAuthLink
                        className={cx('forceLogin')}
                        image="yandex"
                        message={favoritesKeyset.forceLogin()}
                    />
                )}

                {hotelsNode}

                {navigationButtons}
            </Flex>
        );
    }, [
        greyBg,
        handleCopyLinkClick,
        hasHotels,
        selectedCategoryName,
        handleShareClick,
        deviceType,
        shareTokenParams,
        openDeleteFavoriteCategoryWarningModal,
        handleSearchFormSubmit,
        navigationButtons,
        selectedCategoryId,
        hotelsNode,
        isAuth,
        isCategoryView,
        resetCategory,
        isWeb,
    ]);

    const layoutProps: Omit<ILayoutDefaultProps, 'children'> = {
        project: EProjectName.FAVORITES,
        footerType: EFooterProject.ACCOUNT,
        showNavigation: deviceType.isDesktop,
        showHeader: isWeb,
        showFooter: isWeb,
        showTabBar: !isCategoryView,
    };

    if (isLoading && (!permalinksList || !permalinksList.length)) {
        return (
            <LayoutDefault {...layoutProps}>
                <div
                    className={cx('root', deviceMods('root', deviceType), {
                        root_showHotelsList: deviceType.isMobile,
                    })}
                >
                    <div className={cx('spinnerWrapper')}>
                        <Spinner size="xl" />
                    </div>
                </div>
            </LayoutDefault>
        );
    }

    if ((!permalinksList || !permalinksList.length) && !isLoading) {
        return (
            <LayoutDefault {...layoutProps}>
                <EmptyFavorites deviceType={deviceType} />
            </LayoutDefault>
        );
    }

    return (
        <LayoutDefault
            {...layoutProps}
            bgColor={greyBg ? ELayoutBgColor.DARK : undefined}
        >
            <div
                className={cx('root', deviceMods('root', deviceType), {
                    root_showHotelsList: deviceType.isMobile,
                })}
            >
                {(!isCategoryView || deviceType.isDesktop) && (
                    <FavoriteCategories deviceType={deviceType} />
                )}

                {deviceType.isDesktop && desktopContent}

                {deviceType.isMobile && isCategoryView && mobileContent}

                {isOpenDeleteFavoriteCategoryWarningModal && (
                    <DeleteFavoriteCategoryWarningModal
                        isVisible={isOpenDeleteFavoriteCategoryWarningModal}
                        onClose={closeDeleteFavoriteCategoryWarningModal}
                        deviceType={deviceType}
                        onDelete={deleteCategory}
                    />
                )}
            </div>
        </LayoutDefault>
    );
};

export default FavoritesPage;
