import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {matchPath, useLocation} from 'react-router-dom';

import {URLs} from 'constants/urls';
import {PAGE_HOTEL_COUNT} from 'projects/hotels/constants/searchPage';
import {EProjectName} from 'constants/common';

import {INavigationTokens} from 'types/hotels/search/INavigationTokens';
import {EFooterProject} from 'components/Footer/types';

import {
    getSharedFavoriteHotelsActions,
    resetActiveHotel as resetActiveHotelAction,
    setActiveHotel as setActiveHotelAction,
} from 'reducers/favorites/actions';

import {getFavoritesInfoSelector} from 'selectors/favorites/getFavoritesInfo';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import useAddFavoriteHotelClick from 'projects/favorites/utilities/useAddFavoriteHotelClick';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import FavoriteHotelsSearchForm from 'projects/favorites/components/FavoriteHotelsSearchForm/FavoriteHotelsSearchForm';
import HotelsList, {
    EHotelsListSkeletonType,
} from 'projects/hotels/components/HotelsList/HotelsList';
import NavigationButtons from 'projects/hotels/components/NavigationButtons/NavigationButtons';

import cx from './SharedFavoritesPage.scss';

const NAVIGATION_TOKENS: INavigationTokens = {
    nextPage: 'nextPage',
    prevPage: 'prevPage',
    currentPage: 'currentPage',
};

function getOffset(token: string, offset: number): number {
    if (token === NAVIGATION_TOKENS.nextPage) {
        return offset + PAGE_HOTEL_COUNT;
    }

    if (token === NAVIGATION_TOKENS.prevPage) {
        return offset - PAGE_HOTEL_COUNT;
    }

    return offset;
}

interface ISharedFavoritesPageProps {}

const SharedFavoritesPage: React.FC<ISharedFavoritesPageProps> = () => {
    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const location = useLocation();

    const {
        offerSearchParams,
        hotelWithOffersByPermalink,
        permalinksList,
        isError,
        isLoading,
        lastSearchTimestamp,
        offerSearchProgress,
        offset,
        totalHotelCount,
        activeHotelPermalink,
        category,
        shareToken,
        nights,
    } = useSelector(getFavoritesInfoSelector);

    const matchUrl = matchPath(location.pathname, {
        path: URLs.favoritesSharedWithToken,
        exact: true,
        strict: true,
    });

    const matchUrlParams = matchUrl?.params;
    const tokenFromUrl =
        matchUrlParams && 'token' in matchUrlParams
            ? matchUrlParams['token']
            : '';

    const handleSearchFormSubmit = useCallback(() => {
        dispatch(
            getSharedFavoriteHotelsActions.request({
                offset,
                limit: PAGE_HOTEL_COUNT,
                token: shareToken || tokenFromUrl,
            }),
        );
    }, [dispatch, shareToken, tokenFromUrl, offset]);

    useEffect(() => {
        handleSearchFormSubmit();
    }, [handleSearchFormSubmit]);

    const setActiveHotel = useDispatchedAction(setActiveHotelAction, dispatch);
    const resetActiveHotel = useDispatchedAction(
        resetActiveHotelAction,
        dispatch,
    ) as () => void;

    const handleChangeNavigationToken = useCallback(
        (token: string) => {
            dispatch(
                getSharedFavoriteHotelsActions.request({
                    offset: getOffset(token, offset),
                    limit: PAGE_HOTEL_COUNT,
                    token: shareToken || tokenFromUrl,
                }),
            );
        },
        [dispatch, offset, shareToken, tokenFromUrl],
    );

    const handleAddFavoriteHotelClick = useAddFavoriteHotelClick();

    const navigationButtons = useMemo(() => {
        return (
            <NavigationButtons
                className={cx('navigationButtons')}
                isLoading={!offerSearchProgress?.finished || isLoading}
                isEmptyList={!permalinksList?.length}
                navigationTokens={{
                    ...NAVIGATION_TOKENS,
                    prevPage: offset > 0 ? NAVIGATION_TOKENS.prevPage : '',
                    nextPage:
                        totalHotelCount &&
                        offset < totalHotelCount - PAGE_HOTEL_COUNT
                            ? NAVIGATION_TOKENS.nextPage
                            : '',
                }}
                onChangeNavigationToken={handleChangeNavigationToken}
            />
        );
    }, [
        offerSearchProgress,
        permalinksList,
        isLoading,
        handleChangeNavigationToken,
        offset,
        totalHotelCount,
    ]);

    return (
        <LayoutDefault
            project={EProjectName.FAVORITES}
            footerType={EFooterProject.ACCOUNT}
            showNavigation={deviceType.isDesktop}
            bgColor={deviceType.isMobile ? ELayoutBgColor.DARK : undefined}
        >
            <div className={cx('root', deviceMods('root', deviceType))}>
                <Flex flexDirection="column" className={cx('header')}>
                    <Text
                        size={deviceType.isDesktop ? 'xxl' : 'xl'}
                        weight="bold"
                        className={cx('categoryName')}
                    >
                        {category?.name}
                    </Text>

                    <FavoriteHotelsSearchForm
                        deviceType={deviceType}
                        className={cx('searchForm')}
                        onSubmit={handleSearchFormSubmit}
                    />
                </Flex>
                <Flex flexDirection="column">
                    <HotelsList
                        className={cx('hotelsList')}
                        nights={nights || 0}
                        deviceType={deviceType}
                        searchInfoFromProps={
                            {
                                ...offerSearchParams,
                                isError,
                                isLoading,
                                lastSearchTimestamp,
                            } || undefined
                        }
                        hotels={permalinksList || undefined}
                        hotelWithOffersByPermalink={
                            hotelWithOffersByPermalink || undefined
                        }
                        totalHotelsList={PAGE_HOTEL_COUNT}
                        searchIsFinished={
                            offerSearchProgress?.finished || isError
                        }
                        skeletonType={EHotelsListSkeletonType.OFFER_SKELETON}
                        onHoverHotelCard={setActiveHotel}
                        onLeaveHotelCard={resetActiveHotel}
                        activeHotelPermalink={activeHotelPermalink}
                        offerRequestParams={offerSearchParams || undefined}
                        currentPage="favorites"
                        onFavoriteClick={handleAddFavoriteHotelClick}
                    />

                    {!isError && navigationButtons}
                </Flex>
            </div>
        </LayoutDefault>
    );
};

export default SharedFavoritesPage;
