import {useCallback, useMemo} from 'react';
import {useDispatch} from 'react-redux';

import {URLs} from 'constants/urls';

import {
    removeFavoriteHotelAction,
    addFavoriteHotelAction,
    changeHotelsIsFavoriteActions,
} from 'reducers/favorites/actions';
import {setSnackbarInfo} from 'reducers/common/snackbar/actions';

import debounceById from 'utilities/functions/debounceById';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as favoritesKeyset from 'i18n/account-Favorites';

import LinkButton from 'components/LinkButton/LinkButton';
import {IFavoriteClick} from 'projects/hotels/components/HotelsList/HotelsList';

const FAVORITES_CLICK_DEBOUNCE = 2000;

export default function (): (params: IFavoriteClick) => void {
    const dispatch = useDispatch();

    const debouncedReactionRequest = useMemo(() => {
        return debounceById((permalink: string) => {
            dispatch(changeHotelsIsFavoriteActions.request({permalink}));
        }, FAVORITES_CLICK_DEBOUNCE);
    }, [dispatch]);

    const addHotelToFavoritesText = useMemo(() => {
        const favorites = (
            <LinkButton href={URLs.favorites} theme="brand">
                {favoritesKeyset.favorites()}
            </LinkButton>
        );

        return insertJSXIntoKey(favoritesKeyset.addFavoriteHotel)({
            favorites,
        });
    }, []);

    return useCallback(
        ({permalink, isFavorite}: IFavoriteClick): void => {
            if (isFavorite) {
                dispatch(removeFavoriteHotelAction({permalink}));
                dispatch(
                    setSnackbarInfo({
                        lastAction: `${permalink}-remove`,
                        hasCancellation: true,
                        message: favoritesKeyset.removeFavoriteHotel(),
                        cancelAction: 'addFavoriteHotelOnFavoritesPage',
                        cancelActionData: {permalink},
                        page: 'favorites',
                    }),
                );
            } else {
                dispatch(addFavoriteHotelAction({permalink}));
                dispatch(
                    setSnackbarInfo({
                        lastAction: `${permalink}-add`,
                        hasCancellation: false,
                        message: addHotelToFavoritesText,
                        page: 'favorites',
                    }),
                );
            }

            debouncedReactionRequest(permalink, permalink);
        },
        [dispatch, debouncedReactionRequest, addHotelToFavoritesText],
    );
}
