import React, {useMemo} from 'react';

import {
    EHappyPageCrossSaleBlock,
    THappyPageCrossSaleBlock,
} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';
import TOrderType from 'types/common/orders/TOrderType';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';
import HotelsCrossSaleMap from './components/HotelsCrossSaleMap/HotelsCrossSaleMap';
import PromoCodeBlock from './components/PromoCodeBlock/PromoCodeBlock';
import TransportBlocks from './components/TransportBlocks/TransportBlocks';

export interface ICrossSalesProps extends IWithClassName, IWithQaAttributes {
    blocks: THappyPageCrossSaleBlock[];
    orderType: TOrderType;
    onHotelsMount?(): void;
    onAllHotelsClick?(): void;
    onPromoCodeBlockMount?(): void;
    onPromoCodeBlockClick?(): void;
}

const CrossSales: React.FC<ICrossSalesProps> = props => {
    const {
        className,
        blocks,
        orderType,
        onHotelsMount,
        onAllHotelsClick,
        onPromoCodeBlockClick,
        onPromoCodeBlockMount,
    } = props;
    const deviceType = useDeviceType();

    const sortedBlocks = useMemo(
        () => [...blocks].sort((a, b) => a.order - b.order),
        [blocks],
    );

    return (
        <Box
            className={className}
            between={deviceType.isMobile ? 2 : 10}
            {...prepareQaAttributes(props)}
        >
            {sortedBlocks.map((block, index) => {
                switch (block.blockType) {
                    case EHappyPageCrossSaleBlock.HOTEL_CROSS_SALE: {
                        const {
                            uiPayload: {requestParams},
                        } = block;

                        if (!requestParams.adults) {
                            return null;
                        }

                        return (
                            <HotelsCrossSaleMap
                                key={index}
                                hotelCrossSaleInfo={block.uiPayload}
                                deviceType={deviceType}
                                orderType={orderType}
                                onHotelsMount={onHotelsMount}
                                onAllHotelsClick={onAllHotelsClick}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'hotelsCrossSaleMap',
                                })}
                            />
                        );
                    }

                    case EHappyPageCrossSaleBlock.TRANSPORT_CROSS_SALE: {
                        const {transportBlocks} = block.uiPayload;

                        if (!transportBlocks) {
                            return null;
                        }

                        return (
                            <TransportBlocks
                                key={index}
                                regionToLinguistics={
                                    block.uiPayload.regionToLinguistics
                                }
                                blocks={transportBlocks}
                            />
                        );
                    }

                    case EHappyPageCrossSaleBlock.COMMON_PROMO_CODE: {
                        return (
                            <PromoCodeBlock
                                key={index}
                                promoCodeInfo={block.uiPayload}
                                onMount={onPromoCodeBlockMount}
                                onClick={onPromoCodeBlockClick}
                            />
                        );
                    }

                    default: {
                        return null;
                    }
                }
            })}
        </Box>
    );
};

export default React.memo(CrossSales);
