import {FunctionComponent, memo, useEffect} from 'react';
import {useInView} from 'react-intersection-observer';

import {IWithDeviceType} from 'types/withDeviceType';
import {IHotelCrossSaleInfo} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';
import {EGeoObjectType} from 'types/common/Geo';
import TOrderType from 'types/common/orders/TOrderType';

import getSearchHotelsPageYtpRefererByOrderType from './utilities/getSearchHotelsPageYtpRefererByOrderType';
import {getCrossSaleHotelsSearchUrl} from 'projects/avia/utilities/getCrossSaleHotelsSearchUrl';
import {isV3Response} from 'utilities/crossSaleHotels/isV3Response';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import useHotelsCrossSale from 'hooks/useHotelsCrossSale/useHotelsCrossSale';

import CrossSaleMap from 'components/CrossSaleMap/CrossSaleMap';
import CrossSaleMapSkeleton from 'components/SearchHotelsCrossSaleMap/components/CrossSaleMapSkeleton/CrossSaleMapSkeleton';

interface IHotelsCrossSaleMapProps extends IWithDeviceType, IWithQaAttributes {
    hotelCrossSaleInfo: IHotelCrossSaleInfo;
    orderType: TOrderType;
    onHotelsMount?(): void;
    onAllHotelsClick?: () => void;
}

const HotelsCrossSaleMap: FunctionComponent<IHotelsCrossSaleMapProps> =
    props => {
        const {
            orderType,
            hotelCrossSaleInfo,
            deviceType,
            onHotelsMount,
            onAllHotelsClick,
        } = props;
        const {requestParams, regionGeoId} = hotelCrossSaleInfo;

        const [ref, wasInView] = useInView({
            threshold: 0.8,
            triggerOnce: true,
        });

        const {fetch, reset, isError, isLoading, data} = useHotelsCrossSale();

        useEffect(() => {
            onHotelsMount?.();
        }, [onHotelsMount]);

        useEffect(() => {
            fetch({
                ...requestParams,
                totalHotelLimit: 7,
                pointKey: `${EGeoObjectType.City}${regionGeoId}`,
            });

            return (): void => {
                reset();
            };
        }, [fetch, regionGeoId, requestParams, reset]);

        const crossSale = data && (isV3Response(data) ? data.crossSale : data);

        if (isLoading || !wasInView) {
            return (
                <CrossSaleMapSkeleton
                    ref={ref}
                    deviceType={deviceType}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'skeleton',
                    })}
                />
            );
        }

        if (isError || !crossSale) {
            return null;
        }

        return (
            <CrossSaleMap
                ref={ref}
                linkUrl={getCrossSaleHotelsSearchUrl({
                    geoId: crossSale.region.geoId,
                    offerSearchParams: requestParams,
                    ytpReferer:
                        getSearchHotelsPageYtpRefererByOrderType(orderType),
                })}
                crossSaleData={crossSale}
                shadow={deviceType.isMobile ? 'none' : 'default'}
                onLinkClick={onAllHotelsClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'card',
                })}
            />
        );
    };

export default memo(HotelsCrossSaleMap);
