import React, {useEffect} from 'react';
import moment from 'moment';

import {PROMO_CODE_RULES_URL} from 'constants/common';

import {IHappyPageCrossSaleCommonPromoCodeInfo} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {HUMAN_DATE_RU, TIME} from 'utilities/dateUtils/formats';
import {MOSCOW_TIME_ZONE} from 'utilities/dateUtils/timezones';

import * as i18nBlock from 'i18n/happyPage';

import Heading from 'components/Heading/Heading';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import PromoCode, {EPromoCodeColor} from 'components/PromoCode/PromoCode';
import Price from 'components/Price/Price';

import cx from './PromoCodeBlock.scss';

interface IPromoCodeBlockProps {
    promoCodeInfo: IHappyPageCrossSaleCommonPromoCodeInfo;
    onMount?(): void;
    onClick?(): void;
}

const PromoCodeBlock: React.FC<IPromoCodeBlockProps> = ({
    promoCodeInfo: {promoCodeName, discountAmount, minOrderPrice, validTill},
    onMount,
    onClick,
}) => {
    const deviceType = useDeviceType();

    useEffect(() => {
        onMount?.();
    }, [onMount]);

    const titleNode = (
        <Heading className={cx('title')} level={2}>
            {i18nBlock.promoCodeBlockTitleNew()}
        </Heading>
    );

    const validTillNode = (
        <Text size="m">
            {insertJSXIntoKey(i18nBlock.promoCodeBlockValidTill)({
                validTill: moment(validTill)
                    .tz(MOSCOW_TIME_ZONE)
                    .format(`${TIME} ${HUMAN_DATE_RU}`),
                rulesLink: (
                    <Link
                        theme="normal"
                        url={PROMO_CODE_RULES_URL}
                        target="_blank"
                    >
                        {i18nBlock.promoCodeBlockRulesLink()}
                    </Link>
                ),
            })}
        </Text>
    );

    const promoCodeNode = (
        <PromoCode
            className={cx('promoCode')}
            value={promoCodeName}
            color={EPromoCodeColor.BLUE}
        />
    );

    const descriptionNode = (
        <Text className={cx('description')} size="m">
            {insertJSXIntoKey(i18nBlock.promoCodeBlockDescriptionNew)({
                discountAmount: <Price {...discountAmount} />,
                minOrderPrice: <Price {...minOrderPrice} />,
            })}
        </Text>
    );

    const rootClassName = cx('root', deviceMods('root', deviceType));

    if (deviceType.isDesktop) {
        return (
            <Flex
                className={rootClassName}
                flexDirection="column"
                between={5}
                onClick={onClick}
            >
                {titleNode}
                <CardWithDeviceLayout paddingTop={8}>
                    <Flex
                        inline
                        justifyContent="space-between"
                        alignItems="center"
                    >
                        {descriptionNode}
                        {promoCodeNode}
                    </Flex>
                </CardWithDeviceLayout>
                {validTillNode}
            </Flex>
        );
    }

    return (
        <CardWithDeviceLayout
            className={rootClassName}
            paddingTop={5}
            paddingBottom={4}
            shadow="none"
            onClick={onClick}
        >
            {titleNode}
            <Flex flexDirection="column" between={5}>
                {descriptionNode}
                {promoCodeNode}
                {validTillNode}
            </Flex>
        </CardWithDeviceLayout>
    );
};

export default PromoCodeBlock;
