import React, {useMemo} from 'react';

import {IBasicGeoRegion} from 'server/api/HotelsBookAPI/types/IBookOfferResponse';
import {IHappyPageCrossSaleTransportBlock} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {getBlockUrl} from './utilities/getBlockUrl';

import * as i18nBlock from 'i18n/happyPage';

import Heading from 'components/Heading/Heading';
import TransportBlock, {
    ETransportBlockSize,
} from './components/TransportBlock/TransportBlock';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Flex from 'components/Flex/Flex';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';

import cx from './TransportBlocks.scss';

const BLOCK_SIZES = [
    ETransportBlockSize.BIG,
    ETransportBlockSize.BIG,
    ETransportBlockSize.MEDIUM,
    ETransportBlockSize.SMALL,
];

interface ITransportBlocksProps {
    regionToLinguistics: IBasicGeoRegion['linguistics'];
    blocks: IHappyPageCrossSaleTransportBlock[];
}

const TransportBlocks: React.FC<ITransportBlocksProps> = ({
    regionToLinguistics,
    blocks,
}) => {
    const deviceType = useDeviceType();

    const title = useMemo(() => {
        if (!regionToLinguistics) {
            return null;
        }

        const {genitiveCase: regionTo} = regionToLinguistics;

        return i18nBlock.hotelsTransportBlocksTitle({regionTo});
    }, [regionToLinguistics]);

    const blocksNodes = blocks.map(block => (
        <TransportBlock
            key={block.type}
            type={block.type}
            className={cx('block')}
            url={getBlockUrl(block)}
            size={BLOCK_SIZES[blocks.length]}
        />
    ));

    const content = useMemo(() => {
        return (
            <>
                <Heading className={cx('title')} level={2}>
                    {title}
                </Heading>

                {deviceType.isMobile ? (
                    <HorizontalScroller offset={4} className={cx('blocks')}>
                        <Flex
                            inline
                            nowrap
                            between={3}
                            className={cx('blocksWrap')}
                        >
                            {blocksNodes}
                        </Flex>
                    </HorizontalScroller>
                ) : (
                    <div className={cx('blocks')}>{blocksNodes}</div>
                )}
            </>
        );
    }, [blocksNodes, deviceType.isMobile, title]);

    if (!blocks.length) {
        return null;
    }

    return (
        <div className={cx(deviceMods('root', deviceType))}>
            {deviceType.isMobile ? (
                <CardWithDeviceLayout
                    paddingTop={5}
                    paddingBottom={4}
                    shadow="none"
                >
                    {content}
                </CardWithDeviceLayout>
            ) : (
                content
            )}
        </div>
    );
};

export default TransportBlocks;
