import React, {useEffect, useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {TBlockTypes} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {reachGoal} from 'utilities/metrika';
import {getClickMetricaEvent, getMountMetricaEvent} from './utilities';

import * as i18nBlock from 'i18n/happyPage';

import TimerM from 'icons/illustrations/Timer/TimerM';
import RubM from 'icons/illustrations/Rub/RubM';
import StarsM from 'icons/illustrations/Stars/StarsM';
import Text from 'components/Text/Text';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import BlockIcon from '../OrderIcon/BlockIcon';

import cx from './TransportBlock.scss';

export enum ETransportBlockSize {
    SMALL = 'small',
    MEDIUM = 'medium',
    BIG = 'big',
}

const IMAGE_NODE_BY_PROJECT_TYPE = {
    [EProjectName.AVIA]: <TimerM className={cx('image', 'timer')} />,
    [EProjectName.BUSES]: <RubM className={cx('image', 'rub')} />,
    [EProjectName.TRAINS]: <StarsM className={cx('image', 'stars')} />,
};

interface ITransportBlockProps extends IWithClassName {
    type: TBlockTypes;
    url: string;
    size?: ETransportBlockSize;
}

const TransportBlock: React.FC<ITransportBlockProps> = ({
    className,
    type,
    url,
    size,
}) => {
    const deviceType = useDeviceType();

    const clickMetricaEvent = getClickMetricaEvent(type);
    const reachClick = useReachGoal(clickMetricaEvent);

    useEffect(() => {
        const metricaEvent = getMountMetricaEvent(type);

        reachGoal(metricaEvent);
    }, [type]);

    const description = useMemo((): string => {
        switch (type) {
            case EProjectName.AVIA:
                return i18nBlock.hotelsCrossSalesDescriptionAvia();
            case EProjectName.BUSES:
                return i18nBlock.hotelsCrossSalesDescriptionBuses();
            case EProjectName.TRAINS:
                return i18nBlock.hotelsCrossSalesDescriptionTrains();
        }
    }, [type]);
    const linkTitle = useMemo(
        () =>
            type === EProjectName.AVIA
                ? i18nBlock.findFlight()
                : i18nBlock.buyTicket(),
        [type],
    );
    const image = useMemo(() => {
        if (size === ETransportBlockSize.BIG && deviceType.isDesktop) {
            return (
                <>
                    <TimerM className={cx('image', 'timer')} />
                    <RubM className={cx('image', 'rub')} />
                    <StarsM className={cx('image', 'stars')} />
                </>
            );
        }

        return IMAGE_NODE_BY_PROJECT_TYPE[type];
    }, [type, size, deviceType]);

    return (
        <div
            className={cx(
                'root',
                deviceMods('root', deviceType),
                deviceMods('root', deviceType, size),
                className,
            )}
        >
            {image}
            <div className={cx('descriptionWrap')}>
                <BlockIcon type={type} />
                <Text className={cx('description')}>{description}</Text>
            </div>
            <div className={cx('linkWrap')}>
                <ButtonLink
                    target="_blank"
                    className={cx('link')}
                    to={url}
                    size={deviceType.isDesktop ? 's' : 'l'}
                    onClick={reachClick}
                >
                    {linkTitle}
                </ButtonLink>
            </div>
        </div>
    );
};

export default TransportBlock;
