import React, {useCallback} from 'react';
import {Redirect} from 'react-router-dom';

import {URLs} from 'constants/urls';
import {FETCH_ORDER_ERROR} from 'constants/errors/orderErrors';

import {EFooterProject} from 'components/Footer/types';
import {EHttpStatusCodes} from 'types/httpStatusCodes';
import TOrderType from 'types/common/orders/TOrderType';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import {
    IWithDataQaAttribute,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/account-Order';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import NotFoundContent from 'components/NotFound/NotFoundContent/NotFoundContent';
import NotFoundWrapper from 'components/NotFound/NotFoundWrapper/NotFoundWrapper';
import HappyPageLayout from 'components/Layouts/HappyPageLayout/HappyPageLayout';
import ErrorModal from 'components/ErrorModal/ErrorModal';

interface IErrorPageProps extends IWithDataQaAttribute {
    orderType: TOrderType;
    footerType: EFooterProject;
    statusCode?: number;
    orderId: string | undefined;
    onErrorModalMount?(): void;
}

const ErrorPage: React.FC<IErrorPageProps> = ({
    statusCode,
    orderType,
    footerType,
    orderId,
    onErrorModalMount,
    ...rest
}) => {
    const reloadCallback = useCallback(
        (): void => document.location.reload(),
        [],
    );

    if (statusCode === EHttpStatusCodes.NotFound || !orderId) {
        return (
            <NotFoundWrapper>
                <LayoutDefault>
                    <NotFoundContent />
                </LayoutDefault>
            </NotFoundWrapper>
        );
    }

    if (statusCode === EHttpStatusCodes.Forbidden) {
        return <Redirect to={getOrderUrl(orderId)} />;
    }

    return (
        <HappyPageLayout
            orderType={orderType}
            footerType={footerType}
            error={
                <ErrorModal
                    title={i18nBlock.errorModalDotHeader()}
                    text={FETCH_ORDER_ERROR.text}
                    primaryAction={{
                        title: i18nBlock.errorModalDotGoToMainButton(),
                        props: {url: URLs.index},
                        type: 'button-link',
                    }}
                    secondaryAction={{
                        title: i18nBlock.errorModalDotRetryButton(),
                        type: 'button',
                        handler: reloadCallback,
                    }}
                    isVisible
                    onMount={onErrorModalMount}
                    {...prepareQaAttributes(rest)}
                />
            }
        />
    );
};

export default ErrorPage;
