import React, {FC, memo} from 'react';

import TOrderType from 'types/common/orders/TOrderType';
import {IWithClassName} from 'types/withClassName';
import {IHappyPageCrossSale} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {isPromoBlock} from 'projects/happyPage/utilities/crossSale';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import SuccessImage from 'components/Layouts/HappyPageLayout/components/SuccessImage/SuccessImage';
import Container from 'components/Layouts/Container/Container';
import DesktopOrderHeader from 'components/Layouts/HappyPageLayout/components/DesktopOrderHeader/DesktopOrderHeader';
import SuccessText from 'components/Layouts/HappyPageLayout/components/SuccessText/SuccessText';
import Flex from 'components/Flex/Flex';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import CrossSales, {
    ICrossSalesProps,
} from 'projects/happyPage/components/CrossSales/CrossSales';
import OrderId from 'components/Layouts/HappyPageLayout/components/OrderId/OrderId';

import cx from './HappyPageContent.scss';

interface IHappyPageContentProps extends IWithClassName, IWithQaAttributes {
    orderType: TOrderType;
    orderInfo: React.ReactNode;
    orderActions: React.ReactNode;
    crossSale: IHappyPageCrossSale;
    prettyOrderId: string;
    email: string;

    title?: string;
    banner?: React.ReactNode;
    orderHeaderInfo?: React.ReactNode;
    crossSaleProps?: Omit<ICrossSalesProps, 'blocks' | 'orderType'>;
}

const HappyPageContent: FC<IHappyPageContentProps> = props => {
    const {
        className,
        orderType,
        orderInfo,
        orderActions,
        crossSale,
        prettyOrderId,
        email,
        title,
        banner,
        orderHeaderInfo,
        crossSaleProps: {
            className: crossSaleClassName,
            ...crossSaleRestProps
        } = {},
    } = props;

    const deviceType = useDeviceType();

    const hasPromoBlock = crossSale.blocks.some(isPromoBlock);

    return (
        <div className={cx('root', deviceMods('root', deviceType), className)}>
            {deviceType.isMobile && <SuccessImage orderType={orderType} />}

            <Container className={cx('container')}>
                {deviceType.isDesktop ? (
                    <DesktopOrderHeader
                        orderType={orderType}
                        prettyOrderId={prettyOrderId ?? ''}
                        email={email}
                        title={title}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'desktopOrderHeader',
                        })}
                    />
                ) : (
                    <SuccessText
                        className={cx('successText')}
                        title={title}
                        email={email}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'mobileSuccessText',
                        })}
                    />
                )}

                {banner}
                {orderHeaderInfo}

                <div className={cx('orderInfo')}>{orderInfo}</div>

                <Flex
                    flexDirection={deviceType.isMobile ? 'column' : 'row'}
                    justifyContent="flex-end"
                    alignItems={deviceType.isMobile ? 'stretch' : 'center'}
                    above={deviceType.isMobile ? 3 : 5}
                >
                    {orderActions}
                </Flex>

                {deviceType.isMobile && (
                    <OrderId
                        className={cx('orderId')}
                        orderType={orderType}
                        prettyOrderId={prettyOrderId}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'mobileOrderId',
                        })}
                    />
                )}

                {hasPromoBlock && (
                    <AdFoxBanner
                        className={cx('promoBlock')}
                        type={EAdFoxBannerType.Inline}
                        position={EAdFoxBannerPosition.Center}
                    />
                )}
            </Container>

            <CrossSales
                className={cx('crossSales', crossSaleClassName)}
                orderType={orderType}
                blocks={crossSale.blocks}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'crossSales',
                })}
                {...crossSaleRestProps}
            />
        </div>
    );
};

export default memo(HappyPageContent);
