import React, {useEffect} from 'react';
import {Route, Switch} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {URLs} from 'constants/urls';

import {ELoadableReducer} from 'types/common/ELoadableReducer';

import {
    resetOrderInfoAction,
    stopOrderInfoPollingAction,
    stopCreateOrderActions,
    stopOrderPaymentActions,
} from 'reducers/hotels/bookAndPayPage/orderInfo/actions';
import bookAndPayPageReducer from 'reducers/hotels/bookAndPayPage/bookAndPayReducer';
import happyPageReducer from 'reducers/happyPage/reducer';

import getOfferInfoByToken from 'sagas/hotels/bookAndPay/getOfferInfoByToken';
import createOrder from 'sagas/hotels/bookAndPay/createOrder';
import estimateDiscount from 'sagas/hotels/bookAndPay/estimateDiscount';
import pollingOrder from 'sagas/hotels/bookAndPay/pollingOrder';
import startPaymentOrder from 'sagas/hotels/bookAndPay/startPaymentOrder';
import cancelOrder from 'sagas/hotels/bookAndPay/cancelOrder';
import removeOrder from 'sagas/hotels/bookAndPay/removeOrder';
import requestHappyPageSaga from 'sagas/happyPage/orderAndCrossSales/requestHappyPageSaga';

import {loadable} from 'utilities/pageLoadable';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

const BookPage = loadable(
    () => import('projects/hotels/pages/BookPage/BookPageContainer'),
);

const PaymentPage = loadable(
    () => import('projects/hotels/pages/PaymentPage/PaymentPageContainer'),
);

const FinishPaymentPage = loadable(
    () => import('components/Layouts/PaymentFinish/PaymentFinish'),
);

const HappyPage = loadable(
    () => import('projects/hotels/pages/HappyPage/HappyPage'),
);

const BookErrorPage = loadable(
    () => import('projects/hotels/pages/BookErrorPage/BookErrorPage'),
);

const CancelledOrderPage = loadable(
    () => import('projects/hotels/pages/CancelledOrderPage/CancelledOrderPage'),
);

const AuthorizeVoucherPage = loadable(
    () => import('containers/AuthTicket/AuthTicket'),
);

const HotelsStartDeferredPaymentPage = loadable(
    () =>
        import(
            'projects/hotels/pages/HotelsStartDeferredPaymentPage/HotelsStartDeferredPaymentPageContainer'
        ),
);

const NotFoundApp = loadable(
    () => import('components/NotFound/NotFoundPage/NotFoundPage'),
);

interface IHotelsBookAppProps {}

const HotelsBookApp: React.FC<IHotelsBookAppProps> = () => {
    const dispatch = useDispatch();

    useEffect(() => {
        return (): void => {
            dispatch(stopCreateOrderActions());
            dispatch(stopOrderPaymentActions());
            dispatch(stopOrderInfoPollingAction());
            dispatch(resetOrderInfoAction());
        };
    }, [dispatch]);

    return (
        <Switch>
            <Route exact path={URLs.hotelsBook} component={BookPage} />
            <Route path={URLs.hotelsBookPayment} component={PaymentPage} />
            <Route
                path={URLs.hotelsBookFinishPayment}
                component={FinishPaymentPage}
            />
            <Route path={URLs.hotelsBookSuccess} component={HappyPage} />
            <Route path={URLs.hotelsBookError} component={BookErrorPage} />
            <Route
                path={URLs.hotelsCancelledOrder}
                component={CancelledOrderPage}
            />
            <Route
                path={URLs.hotelsAuthorizeVoucher}
                component={AuthorizeVoucherPage}
            />
            <Route
                path={URLs.hotelsBookStartDeferredPayment}
                component={HotelsStartDeferredPaymentPage}
            />
            <Route component={NotFoundApp} />
        </Switch>
    );
};

export default React.memo(
    withReducers([
        [ELoadableReducer.HOTELS_BOOK_AND_PAY, bookAndPayPageReducer],
        [ELoadableReducer.HAPPY_PAGE, happyPageReducer],
    ])(
        withSaga([
            getOfferInfoByToken,
            createOrder,
            estimateDiscount,
            pollingOrder,
            startPaymentOrder,
            cancelOrder,
            removeOrder,
            requestHappyPageSaga,
        ])(HotelsBookApp),
    ),
);
