import {FunctionComponent, ReactNode} from 'react';

import {
    ERoomBedIcon,
    IRoomBedConfigurationItem,
} from 'types/hotels/offer/IHotelRoom';
import {IWithClassName} from 'types/withClassName';

import {CHAR_NBSP} from 'utilities/strings/charCodes';

/* Components */
import SingleBedIcon from 'icons/16/SingleBed';
import DoubleBedIcon from 'icons/16/DoubleBed';

/* Init styles */
import cx from './Bed.scss';

/* Types */
export interface IBedProps extends IWithClassName {
    bed: IRoomBedConfigurationItem;
    canViewBedName?: boolean;
    canViewImage?: boolean;
}

/* Constants */
const MAX_ICON_COUNT = 2;
const ICON_BY_BED: Record<ERoomBedIcon, ReactNode> = {
    [ERoomBedIcon.SINGLE_BED]: <SingleBedIcon className={cx('icon')} />,
    [ERoomBedIcon.DOUBLE_BED]: <DoubleBedIcon className={cx('icon')} />,
};

const getRoomBedIconBlockById = ({
    icon,
    quantity,
}: {
    icon?: ERoomBedIcon;
    quantity: number;
}): ReactNode => {
    const iconNode = icon ? ICON_BY_BED[icon] : null;

    if (!iconNode) {
        return null;
    }

    switch (true) {
        case quantity > MAX_ICON_COUNT: {
            return (
                <>
                    {`${quantity}${CHAR_NBSP}x${CHAR_NBSP}`}
                    {iconNode}
                </>
            );
        }

        case quantity === MAX_ICON_COUNT: {
            return (
                <>
                    {iconNode}
                    {iconNode}
                </>
            );
        }

        default: {
            return iconNode;
        }
    }
};

const Bed: FunctionComponent<IBedProps> = ({
    className,
    bed,
    canViewBedName = true,
    canViewImage = true,
}) => {
    const {icon, quantity, nameInflectedForm} = bed;
    const iconBlockNode =
        canViewImage && getRoomBedIconBlockById({icon, quantity});
    const canUseQuantity = quantity <= MAX_ICON_COUNT;
    const isSingleBedIcon = icon === ERoomBedIcon.SINGLE_BED;

    return (
        <div
            className={cx(
                'root',
                {
                    root_hasSingleBedIcon: isSingleBedIcon,
                },
                className,
            )}
        >
            {Boolean(iconBlockNode) && (
                <div className={cx('iconBlock')}>{iconBlockNode}</div>
            )}
            {canViewBedName && (
                <div className={cx('name')}>
                    {`${
                        canUseQuantity ? `${quantity}${CHAR_NBSP}` : ''
                    }${nameInflectedForm}`}
                </div>
            )}
        </div>
    );
};

export default Bed;
