import React from 'react';

import {
    IBookDeferredPaymentSchedule,
    IBookSearchParams,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {ICancellationInfo} from 'types/hotels/offer/IHotelOfferCancellationInfo';

import {IDevice} from 'reducers/common/commonReducerTypes';

import getDeferredPaymentQuestionsAndAnswers from './utilities/getDeferredPaymentQuestionsAndAnswers';

import * as i18nBlock from 'i18n/hotels-DeferredPaymentScheduleInfo';

import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import Price from 'components/Price/Price';

import DeferredPaymentScheduleTimeline from '../DeferredPaymentScheduleTimeline/DeferredPaymentScheduleTimeline';

import cx from './DeferredPaymentScheduleInfo.scss';

interface IDeferredPaymentScheduleInfoProps {
    deferredPaymentSchedule: IBookDeferredPaymentSchedule;
    cancellationInfo: ICancellationInfo;
    deviceType: IDevice;
    searchParams: IBookSearchParams;
}

const DeferredPaymentScheduleInfo: React.FC<IDeferredPaymentScheduleInfoProps> =
    ({deferredPaymentSchedule, cancellationInfo, deviceType, searchParams}) => {
        const {
            initialPayment: {
                amount: {
                    value: initialPaymentValue,
                    currency: initialPaymentCurrency,
                },
            },
            deferredPayments,
            zeroFirstPayment,
        } = deferredPaymentSchedule;

        const [deferredPayment] = deferredPayments;
        const {penaltyIfUnpaid} = deferredPayment;

        const prepaymentPrice = (
            <Price
                value={initialPaymentValue}
                currency={initialPaymentCurrency}
            />
        );

        const penaltyIfUnpaidPrice = (
            <Price
                value={penaltyIfUnpaid.value}
                currency={penaltyIfUnpaid.currency}
            />
        );

        const deferredPaymentQuestionsAndAnswers =
            getDeferredPaymentQuestionsAndAnswers({
                deferredPaymentSchedule,
                cancellationInfo,
                prepaymentPrice,
                penaltyIfUnpaidPrice,
            });

        return (
            <div className={cx('root')}>
                <Text size={deviceType.isMobile ? 'l' : 'xl'} weight="bold">
                    {zeroFirstPayment
                        ? i18nBlock.howNoPrepaymentWorks()
                        : i18nBlock.howPrepaymentWorks()}
                </Text>

                <DeferredPaymentScheduleTimeline
                    deferredPaymentSchedule={deferredPaymentSchedule}
                    checkinDate={searchParams.startDate}
                    cancellationInfo={cancellationInfo}
                />

                <Separator />

                <div className={cx('questionsAndAnswers')}>
                    {deferredPaymentQuestionsAndAnswers.map(
                        questionAndAnswer => (
                            <>
                                <Text
                                    size="m"
                                    weight="bold"
                                    className={cx('questionAndAnswer')}
                                >
                                    {questionAndAnswer.questionText}
                                </Text>

                                <Text
                                    size="m"
                                    className={cx('questionAndAnswer')}
                                >
                                    {questionAndAnswer.answerText}
                                </Text>
                            </>
                        ),
                    )}
                </div>

                <Separator />

                <Text size="s-inset" className={cx('disclaimer')}>
                    {i18nBlock.byMoscowTime()}
                </Text>
            </div>
        );
    };

export default DeferredPaymentScheduleInfo;
