import React from 'react';

import {IBookDeferredPaymentSchedule} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {
    ERefundType,
    ICancellationInfo,
} from 'types/hotels/offer/IHotelOfferCancellationInfo';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareDeferredPaymentDateTime} from 'projects/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';
import {HUMAN, ROBOT_DATETIME, HUMAN_SHORT} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/hotels-DeferredPaymentScheduleInfo';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';

import BracketWithTextUnderInterval from '../BracketWithTextUnderInterval/BracketWithTextUnderInterval';

import cx from './DeferredPaymentScheduleTimeline.scss';

interface IDeferredPaymentScheduleTimelineProps {
    deferredPaymentSchedule: IBookDeferredPaymentSchedule;
    checkinDate: string;
    cancellationInfo?: ICancellationInfo;
}

const DeferredPaymentScheduleTimeline: React.FC<IDeferredPaymentScheduleTimelineProps> =
    ({deferredPaymentSchedule, checkinDate, cancellationInfo}) => {
        const {
            initialPayment: {
                amount: {
                    value: initialPaymentValue,
                    currency: initialPaymentCurrency,
                },
            },
            deferredPayments,
            zeroFirstPayment,
        } = deferredPaymentSchedule;

        const deviceType = useDeviceType();

        const [deferredPayment] = deferredPayments;
        const {
            amount: {
                value: deferredPaymentValue,
                currency: deferredPaymentCurrency,
            },
            paymentEndsAt,
        } = deferredPayment;

        const paymentPrice = (
            <Price
                value={deferredPaymentValue}
                currency={deferredPaymentCurrency}
                className={cx('paymentPrice')}
            />
        );

        const timelinePoint = <div className={cx('timelinePoint')} />;

        const timeline = <div className={cx('timelineLine')} />;

        const deviceDateFormat = deviceType.isMobile ? HUMAN_SHORT : HUMAN;

        const endOfFreeCancellation =
            cancellationInfo?.refundRules?.[0]?.startsAt ?? null;

        const isEndOfFreeCancellationEndsBeforePaymentEndsAt =
            cancellationInfo?.refundType ===
                ERefundType.REFUNDABLE_WITH_PENALTY &&
            endOfFreeCancellation &&
            prepareDeferredPaymentDateTime(
                endOfFreeCancellation,
                ROBOT_DATETIME,
            ) < prepareDeferredPaymentDateTime(paymentEndsAt, ROBOT_DATETIME);

        const preparedEndOfFreeCancellationDateTime =
            endOfFreeCancellation &&
            prepareDeferredPaymentDateTime(
                endOfFreeCancellation,
                deviceDateFormat,
            );

        const endOfFreeCancellationDateTime =
            preparedEndOfFreeCancellationDateTime &&
            i18nBlock.beforeDate({
                date: preparedEndOfFreeCancellationDateTime,
            });

        return (
            <div className={cx('root', deviceMods('root', deviceType))}>
                <div className={cx('timeForPayment')}>
                    <div className={cx('events')}>
                        <Text
                            size="s-inset"
                            className={cx(
                                'prepayment',
                                zeroFirstPayment && 'prepayment_noPrepayment',
                            )}
                        >
                            {zeroFirstPayment
                                ? i18nBlock.noPrepaymentValue()
                                : insertJSXIntoKey(i18nBlock.prepaymentValue)({
                                      value: (
                                          <Price
                                              value={initialPaymentValue}
                                              currency={initialPaymentCurrency}
                                              className={cx('prepaymentPrice')}
                                          />
                                      ),
                                  })}
                        </Text>
                        <Text
                            size="s-inset"
                            className={cx(
                                'payment',
                                zeroFirstPayment && 'payment_noPrepayment',
                            )}
                        >
                            {zeroFirstPayment
                                ? insertJSXIntoKey(
                                      i18nBlock.noPrepaymentPaymentValue,
                                  )({
                                      value: paymentPrice,
                                  })
                                : insertJSXIntoKey(i18nBlock.paymentValue)({
                                      value: paymentPrice,
                                  })}
                        </Text>
                    </div>
                    <div className={cx('timeline')}>
                        {timelinePoint}
                        {timeline}
                    </div>
                    <div className={cx('dates')}>
                        <Text size="s">{i18nBlock.today()}</Text>
                        <Text size="s">
                            {i18nBlock.beforeDate({
                                date: prepareDeferredPaymentDateTime(
                                    paymentEndsAt,
                                    deviceDateFormat,
                                ),
                            })}
                        </Text>
                    </div>
                    <BracketWithTextUnderInterval
                        text={i18nBlock.onYandexTravel()}
                        bracketColor="badge-secondary"
                    />
                    {isEndOfFreeCancellationEndsBeforePaymentEndsAt && (
                        <div className={cx('freeСancellationPoint')}>
                            <Text size="s-inset">
                                {i18nBlock.freeCancellation()}
                            </Text>
                            <div className={cx('timelinePoint')} />
                            <Text size="s-inset">
                                {endOfFreeCancellationDateTime}
                            </Text>
                        </div>
                    )}
                </div>

                <div className={cx('timeBeforeCheckIn')}>
                    <div className={cx('events')}>
                        <Text size="s-inset" className={cx('checkInHotel')}>
                            {i18nBlock.checkInHotel()}
                        </Text>
                    </div>
                    <div className={cx('timeline')}>
                        {timeline}
                        {timelinePoint}
                    </div>
                    <div className={cx('dates')}>
                        <Text size="s">
                            {prepareDeferredPaymentDateTime(
                                checkinDate,
                                deviceDateFormat,
                            )}
                        </Text>
                    </div>
                </div>
            </div>
        );
    };

export default DeferredPaymentScheduleTimeline;
