import React, {PureComponent, ReactNode} from 'react';
import _get from 'lodash/get';

import {IWithClassName} from 'src/types/withClassName';
import {
    IBookDeferredPaymentSchedule,
    IBookOfferPrice,
    IBookOfferPriceInfo,
    IBookSearchParams,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {
    IOrderGuestsInfo,
    IOrderPayment,
} from 'server/api/HotelsBookAPI/types/IOrder';
import {IWithDeviceType} from 'src/types/withDeviceType';
import {IPromoCampaigns} from 'types/hotels/book/IPromoCampaigns';
import {IPromoCodeApplicationResult} from 'types/hotels/book/IApplyPromoCodes';
import {ICancellationInfo} from 'types/hotels/offer/IHotelOfferCancellationInfo';
import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';
import {EFormGroup, IBookFormValues} from 'types/hotels/book/IBookFormValues';
import {IWithWhiteLabelConfig} from 'types/common/whiteLabel/IWithWhiteLabelConfig';

import {IBookPromoCodesReducer} from 'reducers/hotels/bookAndPayPage/promoCodesInfo/reducer';
import {TUserInfo} from 'reducers/common/commonReducerTypes';

import {getTotalNights} from 'projects/hotels/utilities/calculateTotalNights/calculateTotalNights';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import IPrice from 'utilities/currency/PriceInterface';
import getExtraTaxesLabel from 'projects/hotels/components/BookPriceInfo/utilities/getExtraTaxesLabel';
import {CHAR_MINUS} from 'utilities/strings/charCodes';
import getTotalWithdrawPlusPoints from 'projects/hotels/pages/BookPage/utilities/getTotalWithdrawPlusPoints';
import getTotalTopupPlusPoints from 'projects/hotels/pages/BookPage/utilities/getTotalTopupPlusPoints';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {prepareDeferredPaymentDateTime} from 'projects/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';

import * as i18nBlock from 'i18n/hotels-BookPriceInfo';
import * as i18nPlusModeBlock from 'i18n/components-PlusModeBlock';
import * as i18nHotelOrderBadgeBlock from 'i18n/hotels-HotelOfferBadge';
import * as i18nDefferedPaymentBlock from 'i18n/hotels-DeferredPaymentScheduleInfo';

import Text from 'components/Text/Text';
import Card from 'components/Card/Card';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import Separator from 'components/Separator/Separator';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import PlusModeBlock from 'components/PlusModeBlock/PlusModeBlock';
import StriketroughText from 'components/StriketroughText/StriketroughText';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import BookDeferredPaymentSelect from 'projects/hotels/components/BookDeferredPaymentSelect/BookDeferredPaymentSelect';
import HotelsCancellationInfo, {
    EFormatLabel,
} from 'projects/hotels/components/HotelsCancellationInfo/HotelsCancellationInfo';
import GiftIcon from 'icons/16/Gift';
import CheckCircleIcon from 'icons/16/CheckCircle';
import YandexPlusSignBwS from 'icons/illustrations/YandexPlusSignBw/YandexPlusSignBwS';
import PromoCodes from './components/PromoCodes/PromoCodes';
import PlusInfoBlock from './components/PlusInfo/PlusInfoBlock';
import PriceGroupItem from './components/PriceGroupItem/PriceGroupItem';
import WhiteLabelPartnerBonusInfo from './components/WhiteLabelPartnerBonusInfo/WhiteLabelPartnerBonusInfo';
import FormField from 'components/FormField/FormField';
import Form from 'components/Form/Form';

import cx from './BookPriceInfo.scss';

const HOTEL_CHARGES_TOTALS_PATH = 'priceInfo.hotelCharges.totals';
const HOTEL_CHARGES_NIGHTLY_PATH = 'priceInfo.hotelCharges.nightly';

export interface IBookPriceInfoProps
    extends IWithDeviceType,
        IWithClassName,
        IWithQaAttributes,
        IWithWhiteLabelConfig {
    cancellationInfo: ICancellationInfo;
    priceInfo: IBookOfferPriceInfo;
    searchParams: IBookSearchParams;
    paymentInfo: IOrderPayment;
    appliedPromoCode?: IPromoCodeApplicationResult;
    guestsInfo: IOrderGuestsInfo;
    userInfo: TUserInfo;

    deferredPaymentSchedule?: IBookDeferredPaymentSchedule;
    priceAfterPlusWithdrawBottom?: boolean;

    promoCampaigns?: IPromoCampaigns;
    forwardedRef?: React.Ref<HTMLDivElement>;
    promoCodesInfo?: IBookPromoCodesReducer;
    orderStatus?: string;

    formState: IBookFormValues;
    onApplyPromoCodesClick?: (promoCodes: string[]) => void;
    onResetPromoCodesClick?: () => void;
}

export interface IBookPriceInfoState {
    showPricesByNight: boolean;
}

export enum EBookPriceInfoFieldName {
    PLUS_MODE = 'plusMode',
    BEDS_GROUP_ID = 'bedsGroupId',
    USE_DEFERRED_PAYMENTS = 'useDeferredPayments',
}

class BookPriceInfo extends PureComponent<
    IBookPriceInfoProps,
    IBookPriceInfoState
> {
    static defaultProps = {
        cancellationInfo: {},
        searchParams: {},
        priceInfo: {},
        paymentInfo: {},
        guestsInfo: {},
        className: '',
        codeApplicationResults: [],
    };

    constructor(props: IBookPriceInfoProps) {
        super(props);

        this.state = {
            showPricesByNight: false,
        };
    }

    private toggleNightlyPricesVisibility = (): void => {
        this.setState(({showPricesByNight}) => ({
            showPricesByNight: !showPricesByNight,
        }));
    };

    private getOfferPrice = (): IBookOfferPrice | undefined =>
        this.props?.priceInfo.hotelCharges.totals?.totalPrice;

    private getStriketroughPrice = (): IPrice | undefined => {
        const {priceInfo, promoCodesInfo} = this.props;

        const price = promoCodesInfo?.data?.discountPlusPoints
            ? promoCodesInfo?.data?.originalAmount.value
            : priceInfo.hotelCharges.totals?.strikeThrough?.amount;

        const currency =
            promoCodesInfo?.data?.originalAmount.currency ||
            priceInfo.hotelCharges.totals?.strikeThrough?.currency ||
            CurrencyType.RUB;

        if (!price) return;

        return {
            value: Number(price),
            currency,
        };
    };

    private getOfferPriceWithDiscount = (): IPrice | undefined => {
        const {promoCodesInfo} = this.props;

        return promoCodesInfo?.data?.discountedAmount;
    };

    private getTotalOfferPrice = (): IPrice | undefined => {
        const offerPrice = this.getOfferPrice();
        const offerPriceWithDiscount = this.getOfferPriceWithDiscount();

        if (offerPriceWithDiscount) {
            return offerPriceWithDiscount;
        }

        if (!offerPrice) return;

        return {
            value: Number(offerPrice.amount),
            currency: offerPrice?.currency,
        };
    };

    private getTotalBookPrice = (): IPrice | undefined => {
        const totalOfferPrice = this.getTotalOfferPrice();
        const {useDeferredPayments = false} =
            this.props.formState[EFormGroup.PRICE_INFO] || {};
        const deferredPaymentSchedule = this.getDeferredPaymentSchedule();

        if (useDeferredPayments && deferredPaymentSchedule) {
            const {initialPayment} = deferredPaymentSchedule;

            return initialPayment.amount;
        }

        return totalOfferPrice;
    };

    private getDeferredPaymentSchedule = ():
        | IBookDeferredPaymentSchedule
        | undefined => {
        const {promoCodesInfo, deferredPaymentSchedule} = this.props;

        return promoCodesInfo?.isSuccess
            ? promoCodesInfo?.data?.deferredPaymentSchedule
            : deferredPaymentSchedule;
    };

    /* Render */

    private renderHotelChargeTitle(): React.ReactNode {
        return (
            <div
                className={cx('hotelChargeTitle')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'title',
                })}
            >
                {i18nBlock.title()}
            </div>
        );
    }

    private renderNightsPrice(): React.ReactNode {
        const {
            searchParams: {startDate, endDate},
        } = this.props;
        const {showPricesByNight} = this.state;
        const hotelBasePrice = _get(
            this.props,
            `${HOTEL_CHARGES_TOTALS_PATH}.hotelBase`,
            {},
        );
        const nightlyPrices = _get(this.props, HOTEL_CHARGES_NIGHTLY_PATH, []);
        const nightsCount = getTotalNights(startDate, endDate);

        const nightCountNode = (
            <button
                className={cx('toggleButton')}
                type="button"
                onClick={this.toggleNightlyPricesVisibility}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'nightCountButton',
                })}
            >
                <TextWithIcon
                    size="s"
                    text={i18nBlock.nightCount({count: nightsCount})}
                    iconRight={
                        nightlyPrices.length ? RotatingArrowIcon : undefined
                    }
                    iconRightProps={{
                        rotated: showPricesByNight,
                        width: 12,
                        height: 12,
                    }}
                />
            </button>
        );

        const nightlyPricesNode = (
            <AutoHeightAnimateControl canShowComponent={showPricesByNight}>
                {nightlyPrices.map(
                    (
                        {
                            hotelBase: baseNightlyPrice,
                        }: {hotelBase: IBookOfferPrice},
                        index: number,
                    ) => (
                        <PriceGroupItem
                            key={index}
                            isNestedGroup={true}
                            price={baseNightlyPrice}
                            labelNode={i18nBlock.nightPriceLabel({
                                index: index + 1,
                            })}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: 'nightPrice',
                                key: index,
                            })}
                        />
                    ),
                )}
            </AutoHeightAnimateControl>
        );

        return (
            <div>
                <PriceGroupItem
                    labelNode={nightCountNode}
                    price={hotelBasePrice}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'nightsTotal',
                    })}
                />
                {nightlyPricesNode}
            </div>
        );
    }

    private renderDiscount(): React.ReactNode {
        const discount = this.props?.priceInfo.hotelCharges.totals?.discount;

        if (!discount) return null;

        return (
            <div className={cx('discount')}>
                <PriceGroupItem
                    labelNode={i18nBlock.discountLabel()}
                    priceClassName={cx('discount_success')}
                    price={discount}
                    pricePrefix={CHAR_MINUS}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'discount',
                    })}
                />
            </div>
        );
    }

    private renderTaxiBadge(): React.ReactNode {
        const {promoCampaigns, promoCodesInfo} = this.props;

        if (!promoCampaigns?.taxi2020?.eligible) {
            return null;
        }

        if (
            promoCodesInfo?.data?.promoCampaigns?.taxi2020?.eligible === false
        ) {
            return (
                <Text
                    className={cx('promoNotAvailable')}
                    size="s"
                    tag="div"
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'taxiNotAvailableText',
                    })}
                >
                    {i18nHotelOrderBadgeBlock.taxiNotAvailableNow()}
                </Text>
            );
        }

        return (
            <TextWithIcon
                className={cx('badge')}
                size="s"
                iconLeft={GiftIcon}
                text={i18nHotelOrderBadgeBlock.taxiPromocode()}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'taxiBadge',
                })}
            />
        );
    }

    private renderIncludedTaxes(): React.ReactNode {
        const taxesAndFeesSum = _get(
            this.props,
            `${HOTEL_CHARGES_TOTALS_PATH}.taxesAndFeesSum`,
            {},
        );

        const {amount, currency} = taxesAndFeesSum;

        return (
            Boolean(amount && currency) && (
                <div className={cx('priceWrap')}>
                    <PriceGroupItem
                        price={taxesAndFeesSum}
                        className={cx('includedTax')}
                        labelNode={i18nBlock.hotelTaxes()}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'includedTax',
                        })}
                    />
                </div>
            )
        );
    }

    private renderDefferedPrice(): React.ReactNode {
        const {useDeferredPayments = false} =
            this.props.formState[EFormGroup.PRICE_INFO] || {};
        const payment =
            this.getDeferredPaymentSchedule()?.deferredPayments?.[0];

        if (!useDeferredPayments || !payment) return null;

        return (
            <div className={cx('defferedPrice')}>
                <div>
                    {i18nDefferedPaymentBlock.payBeforeEnd({
                        date: prepareDeferredPaymentDateTime(
                            payment.paymentEndsAt,
                        ),
                    })}
                </div>
                <div className={cx('defferedPriceValue')}>
                    <Price
                        value={payment.amount.value}
                        currency={payment.amount.currency}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'deferredPayment-fullPrice',
                        })}
                    />
                </div>
            </div>
        );
    }

    private renderTotalPrice(): React.ReactNode {
        const isDisabled = this.props.promoCodesInfo?.isLoading;
        const totalPrice = this.getTotalBookPrice();
        const strikeTroughPrice = this.getStriketroughPrice();

        if (!totalPrice) {
            return null;
        }

        return (
            <div className={cx('totalPrice')}>
                <div
                    className={cx('totalPriceLabel')}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'totalPriceLabel',
                    })}
                >
                    {i18nBlock.totalPriceLabelNew()}
                </div>
                <div className={cx('totalPriceValue')}>
                    {strikeTroughPrice && !isDisabled && (
                        <StriketroughText
                            crossed
                            theme="alert"
                            className={cx('totalPriceStriketrough')}
                        >
                            <Price
                                color="secondary"
                                {...strikeTroughPrice}
                                {...prepareQaAttributes({
                                    parent: this.props,
                                    current: 'striketroughPrice',
                                })}
                            />
                        </StriketroughText>
                    )}
                    {!isDisabled && (
                        <Price
                            {...totalPrice}
                            color={strikeTroughPrice ? 'alert' : undefined}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: 'totalPrice',
                            })}
                        />
                    )}
                </div>
            </div>
        );
    }

    private renderPlusModeBlock(): React.ReactNode {
        const {
            promoCodesInfo,
            promoCampaigns,
            priceInfo,
            deferredPaymentSchedule,
            deviceType,
        } = this.props;
        const {useDeferredPayments = false} =
            this.props.formState[EFormGroup.PRICE_INFO] || {};

        const priceAfterPlusWithdraw =
            useDeferredPayments && deferredPaymentSchedule?.zeroFirstPayment
                ? undefined
                : priceInfo.hotelCharges.totals?.priceAfterPlusWithdraw;
        const isLoading = promoCodesInfo?.isLoading;
        const topupPlusPoints = getTotalTopupPlusPoints({
            promoCodesInfo,
            promoCampaigns,
        });
        const withdrawPlusPoints = getTotalWithdrawPlusPoints({
            promoCodesInfo,
            promoCampaigns,
        });

        return (
            <FormField
                name={EBookPriceInfoFieldName.PLUS_MODE}
                deviceType={deviceType}
                control={({input}): ReactNode => (
                    <PlusModeBlock
                        className={cx('plusMode')}
                        topupPlusPoints={topupPlusPoints}
                        withdrawPlusPoints={withdrawPlusPoints}
                        onChangePlusMode={input.onChange}
                        plusMode={input.value}
                        priceAfterPlusWithdraw={priceAfterPlusWithdraw}
                        isLoading={isLoading}
                    />
                )}
            />
        );
    }

    private renderPlusInfo(): React.ReactNode {
        const {formState, promoCodesInfo, promoCampaigns} = this.props;
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            formState[EFormGroup.PRICE_INFO] || {};

        const topupPlusPoints = getTotalTopupPlusPoints({
            promoCodesInfo,
            promoCampaigns,
        });
        const withdrawPlusPoints = getTotalWithdrawPlusPoints({
            promoCodesInfo,
            promoCampaigns,
        });

        return (
            <PlusInfoBlock
                topupPlusPoints={topupPlusPoints}
                withdrawPlusPoints={withdrawPlusPoints}
                plusMode={plusMode}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'plusInfo',
                })}
            />
        );
    }

    private renderPartnerBonusInfo(): React.ReactNode {
        const {promoCampaigns = {}} = this.props;

        if (!promoCampaigns?.whiteLabel?.eligible) {
            return null;
        }

        return (
            <WhiteLabelPartnerBonusInfo
                points={promoCampaigns?.whiteLabel?.points}
            />
        );
    }

    private renderPromoCode(): ReactNode {
        const {
            userInfo,
            deviceType,
            promoCodesInfo,
            appliedPromoCode,
            onApplyPromoCodesClick,
            onResetPromoCodesClick,
        } = this.props;

        if (
            !promoCodesInfo ||
            !onApplyPromoCodesClick ||
            !onResetPromoCodesClick
        ) {
            return null;
        }

        return (
            <PromoCodes
                className={cx('promoCodes')}
                deviceType={deviceType}
                userInfo={userInfo}
                promoCodesInfo={promoCodesInfo}
                appliedPromoCode={appliedPromoCode}
                onApplyClick={onApplyPromoCodesClick}
                onResetClick={onResetPromoCodesClick}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'promoCodes',
                })}
            />
        );
    }

    private renderDeferredPaymentSchedule(): ReactNode {
        const {
            cancellationInfo,
            searchParams,
            deviceType,
            formState,
            promoCodesInfo,
        } = this.props;
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            formState[EFormGroup.PRICE_INFO] || {};
        const totalPrice = this.getTotalOfferPrice();
        const deferredPaymentSchedule = this.getDeferredPaymentSchedule();
        const isPlusWithdraw = plusMode === EYandexPlusApplicationMode.WITHDRAW;

        if (!deferredPaymentSchedule || !totalPrice || isPlusWithdraw) {
            return null;
        }

        return (
            <FormField
                name={EBookPriceInfoFieldName.USE_DEFERRED_PAYMENTS}
                deviceType={deviceType}
                control={({input}): ReactNode => (
                    <>
                        <BookDeferredPaymentSelect
                            deviceType={deviceType}
                            cancellationInfo={cancellationInfo}
                            searchParams={searchParams}
                            useDeferredPayments={input.value}
                            deferredPaymentSchedule={deferredPaymentSchedule}
                            onChangeDeferredPayment={input.onChange}
                            isDisabled={Boolean(promoCodesInfo?.isLoading)}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: 'deferredPayment',
                            })}
                        />
                        <Separator margin={4} />
                    </>
                )}
            />
        );
    }

    private renderRefundableRules(): ReactNode {
        const {cancellationInfo, deferredPaymentSchedule} = this.props;

        return (
            cancellationInfo && (
                <HotelsCancellationInfo
                    className={cx('refundableRules')}
                    labelTextSize="s"
                    labelIconLeft={CheckCircleIcon}
                    labelIconLeftClassName={cx('refundableRulesIcon')}
                    cancellationInfo={cancellationInfo}
                    formatLabel={EFormatLabel.ALWAYS_FULL_TEXT}
                    canRenderOnlyFullyRefundable={true}
                    preventRenderDetailedInfo={true}
                    deferredPaymentSchedule={deferredPaymentSchedule}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'cancellationInfo',
                    })}
                />
            )
        );
    }

    private renderSubmitButton(): React.ReactNode {
        const {promoCodesInfo, deferredPaymentSchedule, formState} = this.props;
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            formState[EFormGroup.PRICE_INFO] || {};
        const {useDeferredPayments = false} =
            this.props.formState[EFormGroup.PRICE_INFO] || {};
        const isDisabled = promoCodesInfo?.isLoading;

        const isNoPrepayment =
            useDeferredPayments && deferredPaymentSchedule?.zeroFirstPayment;
        const buttonText = isNoPrepayment
            ? i18nBlock.submitLabelNoPrepayment()
            : i18nBlock.submitLabel();

        return (
            <Button
                className={cx('submit')}
                type="submit"
                theme={
                    plusMode === EYandexPlusApplicationMode.WITHDRAW
                        ? 'plus'
                        : 'primary'
                }
                size="xl"
                width="max"
                disabled={isDisabled}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'submit',
                })}
            >
                {buttonText}
            </Button>
        );
    }

    private renderExtraTaxes(): React.ReactNode {
        const {
            priceInfo: {extraCharges},
        } = this.props;
        const canRenderExtraTaxes = extraCharges.length > 0;

        return (
            canRenderExtraTaxes && (
                <div className={cx('extraTaxes')}>
                    {i18nBlock.extraTaxesTitle()}
                    {extraCharges.map(({payable, type}, index) => (
                        <PriceGroupItem
                            key={index}
                            price={payable}
                            className={cx('extraCharge')}
                            labelNode={getExtraTaxesLabel(type)}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: 'extraTax',
                                key: index,
                            })}
                        />
                    ))}
                </div>
            )
        );
    }

    private renderPriceAfterPlusWithdraw(): React.ReactNode {
        const {
            formState,
            promoCampaigns,
            promoCodesInfo,
            priceInfo: {hotelCharges},
            priceAfterPlusWithdrawBottom,
            deferredPaymentSchedule,
        } = this.props;
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            formState[EFormGroup.PRICE_INFO] || {};
        const {useDeferredPayments = false} =
            this.props.formState[EFormGroup.PRICE_INFO] || {};
        const priceAfterPlusWithdraw =
            hotelCharges.totals?.priceAfterPlusWithdraw;
        const topupPlusPoints = getTotalTopupPlusPoints({
            promoCodesInfo,
            promoCampaigns,
        });
        const withdrawPlusPoints = getTotalWithdrawPlusPoints({
            promoCodesInfo,
            promoCampaigns,
        });

        if (
            !priceAfterPlusWithdraw ||
            !priceAfterPlusWithdrawBottom ||
            !withdrawPlusPoints ||
            topupPlusPoints > withdrawPlusPoints ||
            (useDeferredPayments && deferredPaymentSchedule?.zeroFirstPayment)
        ) {
            return null;
        }

        return (
            <AutoHeightAnimateControl
                canShowComponent={plusMode === EYandexPlusApplicationMode.TOPUP}
                contentClassName={cx('priceAfterPlusDrawContainer')}
            >
                <Card
                    className={cx('priceAfterPlusDraw')}
                    radius="s"
                    x="2"
                    y="2"
                >
                    <TextWithIcon
                        size="s"
                        iconLeft={YandexPlusSignBwS}
                        iconLeftProps={{
                            width: 16,
                            height: 16,
                        }}
                        text={insertJSXIntoKey(
                            i18nPlusModeBlock.priceAfterPlusDraw,
                        )({
                            price: <Price {...priceAfterPlusWithdraw} />,
                        })}
                    />
                </Card>
            </AutoHeightAnimateControl>
        );
    }

    render(): React.ReactNode {
        const {
            className,
            deviceType,
            forwardedRef,
            formState,
            whiteLabelConfig,
        } = this.props;
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            formState[EFormGroup.PRICE_INFO] || {};
        const isWhiteLabel = Boolean(whiteLabelConfig);

        return (
            <Form.FieldGroup groupId={EFormGroup.PRICE_INFO}>
                <div
                    className={cx(
                        deviceMods('bookPriceInfo', deviceType),
                        className,
                    )}
                    ref={forwardedRef}
                    {...prepareQaAttributes(this.props)}
                >
                    {!isWhiteLabel && this.renderPlusModeBlock()}
                    {this.renderHotelChargeTitle()}
                    {this.renderNightsPrice()}
                    {this.renderDiscount()}
                    {this.renderIncludedTaxes()}
                    {!isWhiteLabel &&
                        plusMode === EYandexPlusApplicationMode.WITHDRAW &&
                        this.renderPlusInfo()}
                    {!isWhiteLabel && this.renderPromoCode()}
                    <Separator margin="4" />
                    {this.renderDeferredPaymentSchedule()}
                    {this.renderPriceAfterPlusWithdraw()}
                    {this.renderTotalPrice()}
                    {this.renderDefferedPrice()}
                    {!isWhiteLabel &&
                        plusMode === EYandexPlusApplicationMode.TOPUP &&
                        this.renderPlusInfo()}
                    {isWhiteLabel && this.renderPartnerBonusInfo()}
                    {this.renderTaxiBadge()}
                    {this.renderSubmitButton()}
                    {this.renderRefundableRules()}
                    {this.renderExtraTaxes()}
                </div>
            </Form.FieldGroup>
        );
    }
}

export default BookPriceInfo;
