import React, {useCallback, useState} from 'react';
import {parse} from 'query-string';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    EPromoCodesApplicationType,
    IPromoCodeApplicationResult,
} from 'types/hotels/book/IApplyPromoCodes';

import {TUserInfo} from 'reducers/common/commonReducerTypes';
import {IBookPromoCodesReducer} from 'reducers/hotels/bookAndPayPage/promoCodesInfo/reducer';

import {useToggle} from 'utilities/hooks/useToggle';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18n/hotels-BookPriceInfo';

import Checkbox from 'components/Checkbox/Checkbox';
import AuthCard, {
    RETURN_PARAM_NAME,
    RETURN_PARAM_VALUE,
} from './components/AuthCard/AuthCard';
import InputControls from './components/InputControls/InputControls';

import cx from './PromoCodes.scss';

interface IPromoCodesProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    userInfo: TUserInfo;
    promoCodesInfo: IBookPromoCodesReducer;
    appliedPromoCode?: IPromoCodeApplicationResult;

    onApplyClick: (promoCodes: string[]) => void;
    onResetClick: () => void;
}

const PromoCodes: React.FC<IPromoCodesProps> = ({
    className,
    userInfo,
    deviceType,
    promoCodesInfo,
    appliedPromoCode,
    onApplyClick,
    onResetClick,
    ...props
}) => {
    const {isLoading, isSuccess} = promoCodesInfo;
    const href = location?.href;

    const isSuccessCode =
        appliedPromoCode?.type === EPromoCodesApplicationType.SUCCESS;
    const isApplySuccess = isSuccess && isSuccessCode;
    const isDisabledControls = isLoading || isApplySuccess;
    const isReturnFromAuth =
        parse(location?.search)[RETURN_PARAM_NAME] === RETURN_PARAM_VALUE;
    const isPromoCodeApplied = Boolean(appliedPromoCode);

    const [inputValue, setInputValue] = useState(appliedPromoCode?.code || '');
    const [isChecked, toggleCheckbox] = useToggle(
        Boolean(
            isAuthUser(userInfo) && (isReturnFromAuth || isPromoCodeApplied),
        ),
    );

    const handleCheckboxChange = useCallback(() => {
        toggleCheckbox();
    }, [toggleCheckbox]);

    /* Render */

    return (
        <div className={className}>
            <Checkbox
                className={cx('checkBox')}
                size={deviceType.isDesktop ? 's' : 'm'}
                label={i18nBlock.iHavePromoCode()}
                checked={isChecked}
                onChange={handleCheckboxChange}
                disabled={isDisabledControls}
                {...prepareQaAttributes({parent: props, current: 'checkbox'})}
            />
            {isChecked &&
                (!isAuthUser(userInfo) && href ? (
                    <AuthCard
                        className={cx('authInfo')}
                        href={href}
                        userInfo={userInfo}
                        {...prepareQaAttributes(props)}
                    />
                ) : (
                    <InputControls
                        className={cx('controls')}
                        deviceType={deviceType}
                        userInfo={userInfo}
                        inputValue={inputValue}
                        promoCodesInfo={promoCodesInfo}
                        appliedPromoCode={appliedPromoCode}
                        onInputChange={setInputValue}
                        onApplyClick={onApplyClick}
                        onResetClick={onResetClick}
                        {...prepareQaAttributes(props)}
                    />
                ))}
        </div>
    );
};

export default PromoCodes;
