import React, {ReactNode} from 'react';

import {
    IBookSearchParams,
    IBookHotelInfo,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithClassName} from 'src/types/withClassName';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    prepareSearchDate,
    prepareCheckinCheckoutTime,
} from 'projects/hotels/utilities/getHotelCheckDates/getHotelCheckDates';
import {getTotalNights} from 'projects/hotels/utilities/calculateTotalNights/calculateTotalNights';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {formatGuests} from 'utilities/formatCount';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-BookPayPage';

import Link from 'components/Link/Link';
import Text from 'components/Text/Text';

import cx from './BookSearchParams.scss';

const SEARCH_PARAMS_QA = 'searchParams';

const renderDateAndTime = (
    date: string,
    time: string,
    {isDesktop}: IDevice,
    qaAttributes: IWithQaAttributes,
): ReactNode => (
    <>
        <div className={cx('date')}>
            <Text
                size={isDesktop ? 'l' : 'm'}
                weight="bold"
                {...prepareQaAttributes(qaAttributes)}
            >
                {prepareSearchDate(date)}
            </Text>
        </div>

        <Text className={cx('text')} size="m">
            {time}
        </Text>
    </>
);

const prepareChangeParamsURL = (
    permalink: number,
    {startDate, endDate, adults, childrenAges}: IBookSearchParams,
): string => {
    let ages = Array(adults).fill(88);

    if (childrenAges && childrenAges.length) {
        ages = ages.concat(childrenAges);
    }

    const parameters = {
        middle_yandex_travel_Date: startDate,
        middle_yandex_travel_Nights: getTotalNights(startDate, endDate),
        middle_yandex_travel_Ages: `${ages}`,
        middle_yandex_travel_utm_source: 'travel-portal',
        middle_yandex_travel_utm_medium: 'change-parameters-link',
    };

    return `https://yandex.ru/profile/${permalink}?ag_dynamic=${encodeURIComponent(
        JSON.stringify(parameters),
    )}`;
};

export interface IBookSearchParamsProps
    extends IWithClassName,
        IWithQaAttributes {
    hotelInfo: IBookHotelInfo;
    searchParams: IBookSearchParams;
    showChangeParamsLink?: boolean;
    showGuestsParams?: boolean;
}

const BookSearchParams: React.FC<IBookSearchParamsProps> = props => {
    const {
        hotelInfo: {checkin, checkout, permalink},
        searchParams: {startDate, endDate, adults, childrenAges},
        showChangeParamsLink = true,
        showGuestsParams = true,
        className,
    } = props;
    const {checkinText, checkoutText} = prepareCheckinCheckoutTime({
        checkin,
        checkout,
    });

    const guestsText = formatGuests(adults, childrenAges);

    const deviceType = useDeviceType();
    const checkinDateQaAttributes = prepareQaAttributes({
        parent: props,
        current: `${SEARCH_PARAMS_QA}-checkin-date`,
    });
    const checkoutDateQaAttributes = prepareQaAttributes({
        parent: props,
        current: `${SEARCH_PARAMS_QA}-checkout-date`,
    });
    const guestsQaAttributes = prepareQaAttributes({
        parent: props,
        current: `${SEARCH_PARAMS_QA}-guests`,
    });

    return (
        <div
            className={cx(
                'content',
                deviceMods('content', deviceType),
                className,
            )}
        >
            <div className={cx('dates')}>
                <div className={cx('checkin')}>
                    {renderDateAndTime(
                        startDate,
                        checkinText,
                        deviceType,
                        checkinDateQaAttributes,
                    )}
                </div>
                <Text
                    className={cx('dash')}
                    size={deviceType.isDesktop ? 'l' : 'm'}
                    weight="bold"
                />
                <div className={cx('checkout')}>
                    {renderDateAndTime(
                        endDate,
                        checkoutText,
                        deviceType,
                        checkoutDateQaAttributes,
                    )}
                </div>
            </div>
            {showGuestsParams && (
                <div className={cx('guests')}>
                    <Text size="m" {...prepareQaAttributes(guestsQaAttributes)}>
                        {guestsText}
                    </Text>
                </div>
            )}
            {showChangeParamsLink && (
                <Text size="m">
                    <Link
                        theme="normal"
                        url={prepareChangeParamsURL(permalink, {
                            startDate,
                            endDate,
                            adults,
                            childrenAges,
                        })}
                        text={i18nBlock.changeSearchParams()}
                    />
                </Text>
            )}
        </div>
    );
};

export default BookSearchParams;
