import React, {PureComponent, ChangeEventHandler} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';
import {IBookOfferBedsGroup} from 'reducers/hotels/bookAndPayPage/offerInfoByToken/types';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-SelectBedsGroups';

import Radiobox, {RadioOptionProps} from 'components/Radiobox/Radiobox';

import cx from './BookSelectBedsGroups.scss';

interface IBookSelectBedsGroupsProps extends IWithClassName {
    bedsGroupId: string;
    bedsGroups: IBookOfferBedsGroup[];
    setBedsGroupId(arg: {
        bedsGroupId: string;
        bedsGroupIsSelectedByUser?: boolean;
    }): void;
}

class BookSelectBedsGroups extends PureComponent<IBookSelectBedsGroupsProps> {
    static defaultProps = {
        className: '',
        bedsGroups: [],
        setBedsGroupId: _noop,
    };

    componentDidMount(): void {
        this.setFirstBedsGroup();
    }

    setFirstBedsGroup(): void {
        const {setBedsGroupId, bedsGroups, bedsGroupId} = this.props;
        const isSingleBedsGroup = this.isSingleBedsGroup();

        if (isSingleBedsGroup === false && bedsGroupId === '0') {
            const id = bedsGroups?.[0]?.id;

            if (id) {
                setBedsGroupId({bedsGroupId: id});
            }
        }
    }

    // Бекенд присылает всегда не более на 1ого варианта на своё усмотрение
    isSingleBedsGroup = (): boolean => {
        const {bedsGroups} = this.props;

        return bedsGroups.length === 1;
    };

    setBedsGroupId: ChangeEventHandler<HTMLInputElement> = ({target}) => {
        const {setBedsGroupId} = this.props;
        const {value} = target;

        setBedsGroupId({bedsGroupId: value, bedsGroupIsSelectedByUser: true});
    };

    mapBedsGroupsToOptions = ({
        description,
        id,
    }: IBookOfferBedsGroup): RadioOptionProps => {
        return {
            value: id,
            label: description,
            className: cx('bedsGroup'),
        };
    };

    renderSelectBedsGroups(): React.ReactNode {
        const {bedsGroups, bedsGroupId} = this.props;

        return (
            <React.Fragment>
                <div className={cx('title')}>{i18nBlock.selectTitle()}</div>
                <Radiobox
                    value={bedsGroupId}
                    onChange={this.setBedsGroupId}
                    options={bedsGroups.map(this.mapBedsGroupsToOptions)}
                />
            </React.Fragment>
        );
    }

    renderSingleBedsGroup(): React.ReactNode {
        const {bedsGroups} = this.props;
        const [firstBedsGroup] = bedsGroups;
        const {description} = firstBedsGroup;

        return (
            <React.Fragment>
                <div className={cx('title')}>{i18nBlock.singleTitle()}</div>
                <div className={cx('bedsGroup', 'bedsGroup_single')}>
                    {description}
                </div>
            </React.Fragment>
        );
    }

    hasBedGroups(): boolean {
        const {bedsGroups} = this.props;

        return bedsGroups.length > 0;
    }

    render(): React.ReactNode {
        const {className} = this.props;
        const isSingleBedsGroup = this.isSingleBedsGroup();

        if (!this.hasBedGroups()) {
            return null;
        }

        return (
            <div
                className={cx('bookSelectBedsGroups', className)}
                {...prepareQaAttributes({
                    current: 'bedsGroups',
                })}
            >
                {isSingleBedsGroup
                    ? this.renderSingleBedsGroup()
                    : this.renderSelectBedsGroups()}
            </div>
        );
    }
}

export default BookSelectBedsGroups;
