import React, {useMemo, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as favoritesKeyset from 'i18n/account-Favorites';

import HeartFilledIconS from 'icons/16/HeartFilled';
import HeartFilledIconM from 'icons/24/HeartFilled';
import HeartFilledIconL from 'icons/36/HeartFilled';
import HeartOutlineIconS from 'icons/16/HeartOutline';
import HeartOutlineIconM from 'icons/24/HeartOutline';
import HeartOutlineIconL from 'icons/36/HeartOutline';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import cx from './Favorites.scss';

const OutlineIcons = {
    s: HeartOutlineIconS,
    m: HeartOutlineIconM,
    l: HeartOutlineIconL,
};

const FilledIcons = {
    s: HeartFilledIconS,
    m: HeartFilledIconM,
    l: HeartFilledIconL,
};

export interface IFavoritesProps extends IWithClassName, IWithDeviceType {
    isFavorite?: boolean;
    onClick?: () => void;
    size?: 's' | 'm' | 'l';
    containerClassName?: string;
}

const Favorites: React.FC<IFavoritesProps> = ({
    className,
    isFavorite,
    onClick,
    size = 's',
    deviceType,
}) => {
    const rootRef = useRef<HTMLDivElement>(null);
    const {
        value: isTooltipVisible,
        setTrue: showTooltip,
        setFalse: hideTooltip,
    } = useBoolean(false);
    const {isDesktop} = useDeviceType();

    const Icon = useMemo(() => {
        if (isFavorite) {
            return FilledIcons[size];
        }

        return OutlineIcons[size];
    }, [isFavorite, size]);

    return (
        <>
            <div
                className={cx(
                    'root',
                    `root_size_${size}`,
                    deviceMods('root', deviceType),
                    className,
                )}
                onClick={onClick}
                onMouseEnter={showTooltip}
                onMouseLeave={hideTooltip}
                ref={rootRef}
                {...prepareQaAttributes('addToFavorite')}
            >
                <Icon
                    className={cx(
                        'heartIcon',
                        isFavorite && 'heartIcon_favorite',
                    )}
                />
            </div>
            <MessageBoxPopup
                anchorRef={rootRef}
                isVisible={isTooltipVisible && isDesktop}
                direction={EPopupDirection.BOTTOM}
                className={cx('tooltip')}
            >
                {isFavorite
                    ? favoritesKeyset.removeFromFavorites()
                    : favoritesKeyset.addToFavorites()}
            </MessageBoxPopup>
        </>
    );
};

export default Favorites;
