import {FunctionComponent, useState, useEffect, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {getUserInfo} from 'selectors/common/userInfoSelector';

import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {
    restoreGreetingClosed,
    storeGreetingClosed,
} from './utilities/canRenderGreeting';

import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import ForceLoginNotAuthUser from './components/ForceLoginNotAuthUser/ForceLoginNotAuthUser';
import ForceLoginAuthUser from './components/ForceLoginAuthUser/ForceLoginAuthUser';

import cx from './ForceLogin.scss';

interface IForceLoginProps extends IWithClassName {
    isBookPage?: boolean;
    isSearchPage?: boolean;
}

const ForceLogin: FunctionComponent<IForceLoginProps> = ({
    className,
    isBookPage,
    isSearchPage,
}) => {
    const restoredIsGreetingClosed = restoreGreetingClosed();

    const userInfo = useSelector(getUserInfo);
    const [canRender, setCanRender] = useState(false);
    const [isGreetingClosed, setGreetingClosed] = useState(
        restoredIsGreetingClosed,
    );

    const handleCloseGreetingClick = useCallback(() => {
        setGreetingClosed(true);
        storeGreetingClosed(true);
    }, [setGreetingClosed]);

    useEffect(() => {
        if (!canRender) {
            setCanRender(true);
        }
    }, [canRender]);

    if (!canRender) {
        return <Skeleton className={cx('skeleton', className)} withAnimation />;
    }

    if (!isAuthUser(userInfo)) {
        return (
            <ForceLoginNotAuthUser
                className={className}
                isBookPage={isBookPage}
                isSearchPage={isSearchPage}
            />
        );
    }

    if (!isGreetingClosed) {
        return (
            <ForceLoginAuthUser
                className={className}
                onCloseClick={handleCloseGreetingClick}
                userInfo={userInfo}
                isBookPage={isBookPage}
            />
        );
    }

    return null;
};

export default ForceLogin;
