import React, {FunctionComponent, useCallback, useMemo} from 'react';

/* Types */
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {EPortalSeedReferer} from 'types/hotels/common/IAttribution';
import {IHotelOffer} from 'types/hotels/offer/IHotelOffer';
import {EActiveHotelSource} from 'types/hotels/hotel/IActiveHotel';

/* Utilities */
import {reachGoal} from 'utilities/metrika';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {
    getSearchParamsForBaseHotelUrl,
    hotelsURLs,
} from 'projects/hotels/utilities/urls';
import {convertHotelOrRoomImages} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';

import * as i18nBlock from 'i18n/hotels-HotelPageRooms';

/* Components */
import Link from 'components/Link/Link';
import HotelImageStub from 'projects/hotels/components/HotelImageStub/HotelImageStub';
import HotelCardLabels from 'projects/hotels/pages/SearchPage/components/HotelCardLabels/HotelCardLabels';
import HotelEmptyOffer from 'projects/hotels/pages/SearchPage/components/HotelEmptyOffer/HotelEmptyOffer';
import HotelInfoSearchCard, {
    ECardUsagePage,
} from 'projects/hotels/components/HotelInfoSearchCard/HotelInfoSearchCard';
import HotelSearchCardOfferSkeleton from 'projects/hotels/pages/SearchPage/components/HotelSearchCardOfferSkeleton/HotelSearchCardOfferSkeleton';
import {EHotelsListSkeletonType} from 'projects/hotels/components/HotelsList/HotelsList';
import ImagesCarousel from 'projects/hotels/components/HotelCard/components/HotelCardDesktop/components/ImagesCarousel/ImagesCarousel';

import HotelFirstOffer from '../HotelFirstOffer/HotelFirstOffer';
import {IHotelCardProps} from '../../HotelCard';

/* Init styles */
import cx from './HotelCardDesktop.scss';

/* Constants */
const HOTEL_IMAGE_SIZE_NAME = 'L';
const BUILDING_ILLUSTRATION_PROPS = {height: 64, width: 64};

const HotelCardDesktop: FunctionComponent<IHotelCardProps> = props => {
    const {
        className,
        hotelWithOffers,
        offerRequestParams,
        nights,
        isActive,
        isLazyImagesCarousel = false,
        onHoverHotel,
        onLeaveHotel,
        onCardClick,
        onFavoriteClick,
        renderOfferSkeleton = false,
        skeletonType = EHotelsListSkeletonType.CARD_SKELETON,
        ...rest
    } = props;

    const {hotel, offers, searchedByUser} = hotelWithOffers;
    const {permalink, images} = hotel;
    const [firstOffer]: IHotelOffer[] = offers || [];
    const {hotelsPercentDiscount, hotelsTopSnippet} = useExperiments();
    const deviceType = useDeviceType();
    const {isWhiteLabel} = useWhiteLabelConfig();

    const convertedImages = useMemo(
        () => convertHotelOrRoomImages(images, HOTEL_IMAGE_SIZE_NAME),
        [images],
    );

    const handleNavigationButtonsClick = useCallback(() => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_IMAGE_SWIPE);
    }, []);

    const handleCardClick = useCallback(() => {
        onCardClick?.(Boolean(searchedByUser));
    }, [onCardClick, searchedByUser]);

    const handleCardContentClick = useCallback(() => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_CARD_HOTEL_CLICK);
        handleCardClick();
    }, [handleCardClick]);

    const handleMouseLeave = useCallback(() => {
        if (onLeaveHotel) {
            onLeaveHotel();
        }
    }, [onLeaveHotel]);
    const handleMouseEnter = useCallback(() => {
        if (onHoverHotel) {
            onHoverHotel({
                permalink,
                source: EActiveHotelSource.LIST,
            });
        }
    }, [onHoverHotel, permalink]);

    const searchParams = getSearchParamsForBaseHotelUrl({
        seed: EPortalSeedReferer.PORTAL_HOTELS_SEARCH,
        hotel: hotelWithOffers.hotel,
        offers: hotelWithOffers.offers,
        offerRequestParams,
    });
    const hotelUrl = hotelsURLs.getHotelUrl(searchParams);

    const offer = useMemo(() => {
        if (
            skeletonType === EHotelsListSkeletonType.OFFER_SKELETON &&
            renderOfferSkeleton
        ) {
            return <HotelSearchCardOfferSkeleton />;
        }

        if (!firstOffer) {
            return (
                <HotelEmptyOffer
                    className={
                        skeletonType === EHotelsListSkeletonType.OFFER_SKELETON
                            ? cx('hotelEmptyOffer')
                            : ''
                    }
                />
            );
        }

        if (firstOffer) {
            return (
                <HotelFirstOffer
                    className={cx('rightContentColumn')}
                    offer={firstOffer}
                    firstOfferUrl={hotelUrl}
                    nights={nights}
                    hotelsPercentDiscount={hotelsPercentDiscount}
                    deviceType={deviceType}
                    badge={firstOffer.badges?.[0]}
                    isWhiteLabel={isWhiteLabel}
                    {...prepareQaAttributes(rest)}
                />
            );
        }
    }, [
        skeletonType,
        renderOfferSkeleton,
        firstOffer,
        hotelUrl,
        nights,
        hotelsPercentDiscount,
        deviceType,
        rest,
    ]);

    return (
        <section
            key={permalink}
            className={cx('hotelCard', className)}
            onMouseLeave={handleMouseLeave}
            onMouseEnter={handleMouseEnter}
            {...prepareQaAttributes(props)}
        >
            {(images?.length || 0) > 0 ? (
                <ImagesCarousel
                    className={cx('imagesSlider')}
                    images={convertedImages}
                    hideArrows={!isActive}
                    isLazy={isLazyImagesCarousel}
                    onNavigationButtonClick={handleNavigationButtonsClick}
                />
            ) : (
                <HotelImageStub
                    className={cx('emptyHotelImage')}
                    captionText={i18nBlock.noImages()}
                    illustrationProps={BUILDING_ILLUSTRATION_PROPS}
                />
            )}

            {hotelsTopSnippet && (
                <HotelCardLabels
                    className={cx('labels')}
                    searchedByUser={searchedByUser}
                />
            )}

            <div className={cx('content')}>
                <Link
                    url={hotelUrl}
                    className={cx('hotelLink')}
                    onClick={handleCardContentClick}
                    target="_blank"
                    rel="noopener noreferrer"
                />
                <div className={cx('hotelInfo')}>
                    <HotelInfoSearchCard
                        hotelUrl={hotelUrl}
                        hotelReviewsUrl={hotelUrl}
                        className={cx('leftContentColumn')}
                        hotel={hotel}
                        cardUsagePage={ECardUsagePage.SEARCH_PAGE}
                        onFavoriteClick={onFavoriteClick}
                        {...prepareQaAttributes(props)}
                    />
                    {offer}
                </div>
            </div>
        </section>
    );
};

export default React.memo(HotelCardDesktop);
