import {PureComponent, ReactNode} from 'react';

import {MIN_PERCENT_TO_SHOW} from 'projects/hotels/constants/discount';

import {
    EHotelBadgeType,
    IHotelOffer,
    IHotelOrOfferBadge,
} from 'types/hotels/offer/IHotelOffer';
import {IWithDeviceType} from 'types/withDeviceType';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {ERefundType} from 'types/hotels/offer/IHotelOfferCancellationInfo';

import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import getWhiteLabelBadge from 'components/WhiteLabelLoyaltyBadge/utilities/getWhiteLabelBadge';
import {isYandexPlusBadge} from 'utilities/hotelOffer/getYandexPlusBadge';

import * as i18nBlock from 'i18n/hotels-SearchHotelCard';

import Flex from 'components/Flex/Flex';
import Price from 'components/Price/Price';
import Label, {ELabelThemeType} from 'components/Label/Label';
import PlusInfoPopup from 'components/PlusInfoPopup/PlusInfoPopup';
import ButtonLink, {IButtonLinkProps} from 'components/ButtonLink/ButtonLink';
import WhiteLabelLoyaltyBadge from 'components/WhiteLabelLoyaltyBadge/WhiteLabelLoyaltyBadge';
import HotelOfferBadges from 'projects/hotels/components/HotelOfferBadges/HotelOfferBadges';
import HotelsCancellationInfo, {
    EFormatLabel,
} from 'projects/hotels/components/HotelsCancellationInfo/HotelsCancellationInfo';
import YandexEdaPromo from 'projects/hotels/components/YandexEdaPromo/YandexEdaPromo';
import DiscountInfo from 'projects/hotels/components/HotelCard/components/DiscountInfo/DiscountInfo';

import cx from './HotelFirstOffer.scss';

interface IHotelFirstOfferProps extends IWithDeviceType, IWithQaAttributes {
    className?: string;
    firstOfferUrl: string;
    offer: IHotelOffer;
    hotelsPercentDiscount?: boolean;
    nights: number;
    onOperatorButtonClick?: () => void;
    badge?: IHotelOrOfferBadge;
    isWhiteLabel?: boolean;
}

class HotelFirstOffer extends PureComponent<IHotelFirstOfferProps> {
    private handleOperatorButtonClick = (): void => {
        const {onOperatorButtonClick} = this.props;

        reachGoal(EHotelsGoal.SEARCH_PAGE_CARD_OPERATOR_BUTTON_CLICK);
        onOperatorButtonClick?.();
    };

    private getLinkProps(): Pick<IButtonLinkProps, 'target' | 'size'> &
        (Pick<IButtonLinkProps, 'to'> | Pick<IButtonLinkProps, 'url'>) {
        const {
            firstOfferUrl,
            deviceType: {isMobile},
        } = this.props;

        if (isMobile) {
            return {
                to: firstOfferUrl,
                size: 'l',
                target: '_self',
            };
        }

        return {
            url: firstOfferUrl,
            size: 'm-inset',
            target: '_blank',
        };
    }

    private renderPrice(): ReactNode {
        const {
            offer: {price, discountInfo},
            hotelsPercentDiscount,
        } = this.props;
        const hasDiscount = discountInfo?.strikethroughPrice;
        const showPercentage =
            hotelsPercentDiscount &&
            discountInfo?.percent &&
            discountInfo.percent >= MIN_PERCENT_TO_SHOW;

        return (
            <Price
                color={hasDiscount && !showPercentage ? 'alert' : undefined}
                className={cx('firstOfferPrice')}
                {...price}
            />
        );
    }

    private renderDiscount(): ReactNode {
        const {
            offer: {discountInfo},
            hotelsPercentDiscount,
        } = this.props;

        if (!discountInfo?.strikethroughPrice) return null;

        return (
            <DiscountInfo
                discountInfo={discountInfo}
                hotelsPercentDiscount={hotelsPercentDiscount}
            />
        );
    }

    private renderOperatorButtonLink(): ReactNode {
        return (
            <ButtonLink
                className={cx('firstOfferSelectButton')}
                theme="primary"
                rel="noopener noreferrer"
                onClick={this.handleOperatorButtonClick}
                {...this.getLinkProps()}
            >
                {i18nBlock.selectButton()}
            </ButtonLink>
        );
    }

    private renderCancellationInfo(): ReactNode {
        const {
            offer: {cancellationInfo},
            deviceType: {isMobile},
        } = this.props;

        const label = cancellationInfo && (
            <HotelsCancellationInfo
                cancellationInfo={cancellationInfo}
                labelTextSize="s"
                labelTextColor={isMobile ? 'primary' : undefined}
                labelTextWeight={isMobile ? 'medium' : 'normal'}
                preventRenderDetailedInfo
                canRenderOnlyFullyRefundable
                formatLabel={EFormatLabel.ONLY_REFUND_TEXT}
                {...prepareQaAttributes('cancellationInfo')}
            />
        );

        if (!isMobile) {
            return label;
        }

        const hideNonRefundableBadge =
            cancellationInfo &&
            cancellationInfo.refundType !== ERefundType.FULLY_REFUNDABLE;

        if (hideNonRefundableBadge || !label) {
            return null;
        }

        return (
            <Label theme={ELabelThemeType.NEUTRAL} size="m">
                {label}
            </Label>
        );
    }

    private renderPlusOffer(): ReactNode {
        const {
            offer: {offerYandexPlusInfo},
            deviceType: {isMobile},
            badge,
        } = this.props;

        if (!offerYandexPlusInfo) {
            return null;
        }

        const showButtonTextFromBadge = isYandexPlusBadge(badge);

        return (
            <div className={cx('promoItem')}>
                <PlusInfoPopup
                    labelSize={isMobile ? 'm' : 's-inset'}
                    labelDescriptionType={isMobile ? 'full' : 'short'}
                    title={badge?.additionalPromoInfo?.title}
                    description={badge?.additionalPromoInfo?.text}
                    linkText={badge?.additionalPromoInfo?.link?.text}
                    linkUrl={badge?.additionalPromoInfo?.link?.url}
                    buttonText={
                        showButtonTextFromBadge ? badge?.text : undefined
                    }
                    badgeId={badge?.id}
                    {...offerYandexPlusInfo}
                />
            </div>
        );
    }

    private renderYandexEdaPromo(): ReactNode {
        const {
            offer: {badges},
        } = this.props;

        const yandexEdaBadge = badges?.find(
            badge => badge.id === EHotelBadgeType.YANDEX_EDA,
        );

        return (
            yandexEdaBadge && (
                <div className={cx('promoItem')}>
                    <YandexEdaPromo badge={yandexEdaBadge} />
                </div>
            )
        );
    }

    private renderFeatures(): ReactNode {
        const {
            offer: {mealType, cancellationInfo},
            deviceType: {isMobile},
        } = this.props;
        const hasMeal = mealType && mealType.id !== 'RO';
        const hasFullyRefundable =
            cancellationInfo &&
            cancellationInfo.refundType === ERefundType.FULLY_REFUNDABLE;

        return (
            (hasFullyRefundable || hasMeal) && (
                <div className={cx('offerFeatures')}>
                    {this.renderCancellationInfo()}
                    {hasMeal &&
                        (isMobile ? (
                            <Label
                                className={cx('offerFeature')}
                                size="m"
                                theme={ELabelThemeType.NEUTRAL}
                            >
                                {mealType?.name}
                            </Label>
                        ) : (
                            <div className={cx('offerFeature')}>
                                {mealType?.name}
                            </div>
                        ))}
                </div>
            )
        );
    }

    private renderWhiteLabelBadge(): ReactNode {
        const {offer} = this.props;

        if (offer.badges === undefined) {
            return null;
        }

        const badge = getWhiteLabelBadge(offer.badges);

        if (badge === undefined) {
            return null;
        }

        return <WhiteLabelLoyaltyBadge badge={badge} />;
    }

    private renderBadges(): ReactNode {
        const {offer} = this.props;

        return (
            <HotelOfferBadges
                className={cx('badges')}
                badgeClassName={cx('badge')}
                badges={offer.badges}
                isYandexOffer={offer.yandexOffer}
                above={1}
            />
        );
    }

    renderTotalNights(): ReactNode {
        const {
            nights,
            deviceType: {isMobile},
            ...rest
        } = this.props;
        const mobileText = i18nBlock.nightsCountExpMobile({
            nightsCount: nights,
        });

        return (
            Boolean(nights) && (
                <div
                    className={cx('nightsCount')}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'nightsCount',
                    })}
                >
                    {isMobile
                        ? mobileText
                        : i18nBlock.nightsCount({
                              nightsCount: nights,
                          })}
                </div>
            )
        );
    }

    private renderDesktopOffer(): ReactNode {
        const {isWhiteLabel} = this.props;

        return (
            <>
                <div className={cx('desktopPriceContainer')}>
                    <div>
                        <Flex
                            flexDirection="column"
                            flexWrap="wrap"
                            alignItems="flex-start"
                            between={2}
                            inline
                        >
                            {this.renderPrice()}
                            {this.renderDiscount()}
                        </Flex>
                        {this.renderTotalNights()}
                    </div>
                    {this.renderOperatorButtonLink()}
                </div>
                {isWhiteLabel
                    ? this.renderWhiteLabelBadge()
                    : this.renderPlusOffer()}
                {this.renderYandexEdaPromo()}
                {this.renderBadges()}
                {this.renderFeatures()}
            </>
        );
    }

    private renderMobileOffer(): ReactNode {
        const {isWhiteLabel} = this.props;
        const centerPrice =
            !this.props.offer.offerYandexPlusInfo &&
            !this.props.offer.discountInfo;

        return (
            <>
                {this.renderFeatures()}
                {this.renderBadges()}
                <div
                    className={cx('mobilePriceContainer', {
                        mobilePriceContainer_center: centerPrice,
                    })}
                >
                    <div className={cx('mobilePriceContainerOperator')}>
                        <Flex
                            className={cx('mobilePriceBlock')}
                            alignItems="flex-end"
                        >
                            <Flex
                                flexDirection="column"
                                flexWrap="wrap"
                                alignItems="flex-start"
                                between={0}
                                inline
                            >
                                {this.renderDiscount()}
                                {this.renderPrice()}
                            </Flex>
                            {this.renderTotalNights()}
                        </Flex>
                        {isWhiteLabel
                            ? this.renderWhiteLabelBadge()
                            : this.renderPlusOffer()}
                    </div>
                    {this.renderOperatorButtonLink()}
                </div>
                {this.renderYandexEdaPromo()}
            </>
        );
    }

    render(): ReactNode {
        const {offer, className, deviceType} = this.props;

        if (offer) {
            return (
                <section
                    className={cx(
                        className,
                        'firstOffer',
                        deviceMods('firstOffer', deviceType),
                    )}
                >
                    {deviceType.isMobile
                        ? this.renderMobileOffer()
                        : this.renderDesktopOffer()}
                </section>
            );
        }

        return null;
    }
}

export default HotelFirstOffer;
