import {FunctionComponent, useCallback, MouseEvent} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';
import {IHotelWithMinPrice} from 'types/hotels/hotel/IHotelWithMinPrice';
import {
    IHotelPageQueryParams,
    ISearchPageQueryParams,
} from 'types/hotels/common/IQueryParams';
import {TLinkTarget} from 'types/common/TLinkTarget';

import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useQueryByLocation} from 'utilities/hooks/useQueryByLocation';
import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import {ECardUsagePage} from 'projects/hotels/components/HotelInfoSearchCard/HotelInfoSearchCard';
import HotelCardWihMinPriceMobile from './components/HotelCardWithMinPriceMobile/HotelCardWithMinPriceMobile';
import HotelCardWithMinPriceDesktop from './components/HotelCardWithMinPriceDesktop/HotelCardWithMinPriceDesktop';

export interface IHotelCardWithMinPriceProps
    extends IWithQaAttributes,
        IWithClassName {
    region: IGeoRegion;
    hotelInfoClassName?: string;
    hotelWithMinPrice: IHotelWithMinPrice;
    searchParams?: Partial<IHotelPageQueryParams | ISearchPageQueryParams>;
    cardUsagePage: ECardUsagePage;
    isLink?: boolean;
    linkTarget?: TLinkTarget;
    onClickHotelContent?: (hotelWithMinPrice: IHotelWithMinPrice) => void;
}

export interface ICommonHotelCardWithMinPriceProps
    extends Omit<
        IHotelCardWithMinPriceProps,
        'searchParams' | 'onClickHotelContent' | 'isLink'
    > {
    hotelUrl: string;
    hotelReviewsUrl: string;
    onClick?: (e: MouseEvent) => void;
}

const HotelCardWihMinPrice: FunctionComponent<IHotelCardWithMinPriceProps> =
    props => {
        const {searchParams, onClickHotelContent, isLink, ...rest} = props;
        const {hotelSlug} = props.hotelWithMinPrice.hotel;

        const {isDesktop} = useDeviceType();
        const queryByLocation = useQueryByLocation();

        const hotelUrl = hotelsURLs.getHotelUrl(
            {hotelSlug},
            queryByLocation,
            searchParams,
        );

        const handleCardClick = useCallback(
            (e: MouseEvent) => {
                if (!isLink) {
                    e.preventDefault();
                }

                onClickHotelContent?.(props.hotelWithMinPrice);
            },
            [onClickHotelContent, isLink, props.hotelWithMinPrice],
        );

        return isDesktop ? (
            <HotelCardWithMinPriceDesktop
                hotelUrl={hotelUrl}
                hotelReviewsUrl={hotelUrl}
                onClick={handleCardClick}
                {...rest}
            />
        ) : (
            <HotelCardWihMinPriceMobile
                hotelUrl={hotelUrl}
                hotelReviewsUrl={hotelUrl}
                onClick={handleCardClick}
                {...rest}
            />
        );
    };

export default HotelCardWihMinPrice;
