import {FunctionComponent, ReactNode, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {YANDEX_PLUS_URL} from 'constants/urls/plus';

import {IWithClassName} from 'types/withClassName';
import {TLinkTarget} from 'types/common/TLinkTarget';
import {IHotelOrOfferBadge} from 'types/hotels/offer/IHotelOffer';

import {getUserPlusInfo} from 'selectors/common/userInfoSelector';

import IPrice from 'utilities/currency/PriceInterface';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {useMobile} from 'utilities/hooks/useMobile';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';
import getPlusBadgeButtonText from '../../utilities/getPlusBadgeButtonText';

import * as i18nBlock from 'i18n/hotels-HotelMinPriceOffer';
import * as i18nPlusBlock from 'i18n/components-PlusInfoPopup';
import * as i18nStaticBlock from 'i18n/hotels-seoPage-plusInfo';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import PlusInfoPopup from 'components/PlusInfoPopup/PlusInfoPopup';
import HotelOfferBadges from 'projects/hotels/components/HotelOfferBadges/HotelOfferBadges';

import WhiteLabelConfigContext from 'contexts/WhiteLabelConfigContext';

import cx from './HotelMinPriceOffer.scss';

export interface IHotelMinPriceOfferProps
    extends IWithClassName,
        IWithQaAttributes {
    minPrice: IPrice;
    hotelUrl: string;
    badges?: IHotelOrOfferBadge[];
    nights?: number;
    hasOfferYandexPlus?: boolean;
    linkTarget?: TLinkTarget;
    regionSlug?: string;
}

const HotelMinPriceOffer: FunctionComponent<IHotelMinPriceOfferProps> =
    props => {
        const {isDesktop} = useDeviceType();
        const {
            className,
            minPrice,
            hotelUrl,
            nights,
            badges,
            hasOfferYandexPlus,
            linkTarget = isDesktop ? '_blank' : '_self',
            regionSlug,
        } = props;

        const isMobile = useMobile();
        const userInfo = useUserInfo();
        const isAuth = isAuthUser(userInfo);
        const userPlusInfo = useSelector(getUserPlusInfo);
        const hasPlus = userPlusInfo.hasPlus;
        const experiments = useExperiments();

        const onPlusLinkClick = useCallback(e => {
            e.stopPropagation();
        }, []);

        const plusButtonText = getPlusBadgeButtonText(experiments, regionSlug);

        const plusInfoPopup = (
            <PlusInfoPopup
                labelSize={isMobile ? 'm' : 's-inset'}
                labelDescriptionType={isMobile ? 'full' : 'short'}
                title={i18nPlusBlock.header()}
                description={
                    isAuth && hasPlus
                        ? i18nStaticBlock.description()
                        : i18nStaticBlock.descriptionNotPlusOrAuth()
                }
                linkUrl={YANDEX_PLUS_URL}
                linkText={i18nPlusBlock.moreInfoLink()}
                onPlusLinkClick={onPlusLinkClick}
                buttonText={plusButtonText}
            ></PlusInfoPopup>
        );

        return (
            <WhiteLabelConfigContext.Consumer>
                {(config): ReactNode => (
                    <Box between={isDesktop ? 3 : 2} className={className}>
                        {!isDesktop && (
                            <HotelOfferBadges
                                className={cx('badges')}
                                badges={badges}
                            />
                        )}
                        <Flex justifyContent="space-between" between="2" inline>
                            <div>
                                <Text
                                    className={cx('minPrice')}
                                    size="xl"
                                    tag="div"
                                >
                                    {insertJSXIntoKey(i18nBlock.minPrice)({
                                        priceNode: (
                                            <Price
                                                currency={minPrice.currency}
                                                value={minPrice.value}
                                            />
                                        ),
                                    })}
                                </Text>
                                <Text size="s" tag="div">
                                    {!nights || nights === 1
                                        ? i18nBlock.priceLabel()
                                        : i18nBlock.daysPriceLabel({nights})}
                                </Text>
                            </div>
                            <ButtonLink
                                size={isDesktop ? 'm-inset' : 'l'}
                                target={linkTarget}
                                rel="noopener noreferrer"
                                theme="primary"
                                url={hotelUrl}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'buyButton',
                                })}
                            >
                                {i18nBlock.shortSelectDatesLink()}
                            </ButtonLink>
                        </Flex>
                        {!config && hasOfferYandexPlus && plusInfoPopup}
                        {isDesktop && (
                            <HotelOfferBadges
                                className={cx('badges')}
                                badges={badges}
                            />
                        )}
                    </Box>
                )}
            </WhiteLabelConfigContext.Consumer>
        );
    };

export default HotelMinPriceOffer;
