import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EHotelBanner} from 'server/api/HotelSearchAPI/types/IOffersInfo';
import {EToggler} from 'types/common/togglers/EToggler';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import DirectBanner from 'projects/hotels/components/HotelOfferBanner/components/DirectBanner/DirectBanner';
import MirCashbackBanner from 'projects/hotels/components/MirCashbackBanner/MirCashbackBanner';

import cx from './HotelOfferBanner.scss';

export interface IHotelOfferBadgeProps
    extends IWithClassName,
        IWithQaAttributes {
    bannerType: EHotelBanner | undefined;
    hotelName: string | undefined;
    mirBannerClassName?: string;
}

const HotelOfferBanner: React.FC<IHotelOfferBadgeProps> = props => {
    const {className, bannerType, hotelName = '', mirBannerClassName} = props;

    const deviceType = useDeviceType();

    const bannerClassName = cx('banner', deviceMods('banner', deviceType));

    const badgeContent = useMemo(() => {
        const isPossibleMirCashbackBanner =
            bannerType === EHotelBanner.POSSIBLE_MIR_CASHBACK;
        const isMirCashbackBanner = bannerType === EHotelBanner.MIR_CASHBACK;

        if (isPossibleMirCashbackBanner || isMirCashbackBanner) {
            return (
                <MirCashbackBanner
                    type={
                        isPossibleMirCashbackBanner
                            ? EToggler.MIR_POSSIBLE_CASHBACK_HOTEL
                            : EToggler.MIR_CASHBACK_HOTEL
                    }
                    className={cx(
                        deviceMods('mirCashbackBanner', deviceType),
                        mirBannerClassName,
                    )}
                    {...prepareQaAttributes(props)}
                />
            );
        }

        if (bannerType === EHotelBanner.HOTEL_DIRECT) {
            return (
                <DirectBanner
                    className={bannerClassName}
                    hotelName={hotelName}
                    {...prepareQaAttributes(props)}
                />
            );
        }

        return null;
    }, [
        bannerClassName,
        bannerType,
        hotelName,
        deviceType,
        mirBannerClassName,
    ]);

    if (!badgeContent) {
        return null;
    }

    return (
        <div className={cx('root', deviceMods('root', deviceType), className)}>
            {badgeContent}
        </div>
    );
};

export default HotelOfferBanner;
