import React, {useCallback, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {
    REVIEWS_LOAD_COUNT,
    REVIEWS_SORTING_INFO,
    SET_REVIEW_REACTION_DEBOUNCE,
} from './constants';
import {
    HOTEL_CREATE_REVIEW_ANCHOR,
    HOTEL_REVIEWS_ANCHOR,
} from 'constants/urls/anchors';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IKeyPhrase, ITextReview} from 'types/hotels/hotel/IHotelTextReview';
import {IHotelReviewReactionPayload} from 'reducers/hotels/hotelPage/reviews/list/types';
import {isETextReviewRankingType} from 'server/api/HotelSearchAPI/types/ETextReviewRankingType';
import {TSortId} from 'types/hotels/search/ISortInfo';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {
    filterHotelReviewsActions,
    getHotelReviewsActions,
    setHotelReviewInfoReactionAction,
    setHotelReviewReactionRequestActions,
    sortingHotelReviewsAction,
} from 'reducers/hotels/hotelPage/reviews/list/actions';

import hotelReviewsSelector from './selectors/hotelReviewsSelector';

import debounceById from 'utilities/functions/debounceById';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';
import scrollContainerToNode from 'utilities/dom/scrollContainerToNode';
import {useIsAuth} from 'utilities/hooks/useIsAuth';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';

import * as i18nBlock from 'i18n/hotels-HotelReviews';
import * as i18nCommonBlock from 'i18n/hotels-Common';

import Card from 'components/Card/Card';
import LinkButton from 'components/LinkButton/LinkButton';
import Separator from 'components/Separator/Separator';
import Spinner from 'components/Spinner/Spinner';
import TextHotelReviewsSkeleton from './components/TextHotelReviewsSkeleton/TextHotelReviewsSkeleton';
import HotelReviewWithReaction from './components/TextHotelReview/HotelReviewWithReaction';
import KeyPhraseCheckButton from './components/KeyPhraseCheckButton/KeyPhraseCheckButton';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Flex from 'components/Flex/Flex';
import UserReview from './components/UserReview/UserReview';
import Rating from 'components/Rating/Rating';
import Anchor from 'components/Anchor/Anchor';
import HotelsSortView from 'projects/hotels/components/HotelsSort/components/View';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './HotelReviews.scss';

interface IHotelReviewsProps extends IWithClassName, IWithDeviceType {
    maxReviewRenderCount?: number;
    reviewsRef?: React.RefObject<HTMLDivElement>;
}

const HotelReviews: React.FC<IHotelReviewsProps> = props => {
    const {className, deviceType, reviewsRef, maxReviewRenderCount} = props;

    const {hotelReviews, rating} = useSelector(hotelReviewsSelector);
    const isAuth = useIsAuth();

    const dispatch = useDispatch();
    const getHotelReviews = useDispatchedAction(
        getHotelReviewsActions.request,
        dispatch,
    );
    const filterHotelReviews = useDispatchedAction(
        filterHotelReviewsActions,
        dispatch,
    );
    const setHotelReviewReaction = useDispatchedAction(
        setHotelReviewInfoReactionAction,
        dispatch,
    );
    const setHotelReviewReactionRequest = useDispatchedAction(
        setHotelReviewReactionRequestActions.request,
        dispatch,
    );
    const sortingHotelReviews = useDispatchedAction(
        sortingHotelReviewsAction,
        dispatch,
    );

    const {isWeb} = usePlatform();

    const debouncedReactionRequest = useMemo(() => {
        return debounceById((reactionPayload: IHotelReviewReactionPayload) => {
            setHotelReviewReactionRequest(reactionPayload);
        }, SET_REVIEW_REACTION_DEBOUNCE);
    }, [setHotelReviewReactionRequest]);

    const keyPhrasesRef = useRef<HTMLDivElement>(null);

    const handleKeyPhraseClick = useCallback(
        (keyPhraseEvent: string, element: HTMLElement): void => {
            const {activeKeyPhrase} = hotelReviews;

            const keysPhrase =
                activeKeyPhrase === keyPhraseEvent ? undefined : keyPhraseEvent;

            if (deviceType.isMobile && keyPhrasesRef.current) {
                scrollContainerToNode(keyPhrasesRef.current, element, {
                    horizontal: true,
                    horizontalPadding: 16,
                });
            }

            filterHotelReviews({
                textReviewLimit: REVIEWS_LOAD_COUNT,
                keyPhraseFilter: keysPhrase,
            });

            if (keysPhrase) {
                reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_TAG_CLICK, {
                    hotels: {activeKeyPhrase: keysPhrase},
                });
            }
        },
        [deviceType, filterHotelReviews, hotelReviews],
    );

    const handleReviewsButtonClick = useCallback((): void => {
        const {
            data,
            isLoadingList: isLoading,
            activeKeyPhrase,
            currentSortingOption,
        } = hotelReviews;

        if (!isLoading && data.textReviews.length < data.totalTextReviewCount) {
            getHotelReviews({
                textReviewOffset: data.textReviews.length,
                textReviewLimit: REVIEWS_LOAD_COUNT,
                keyPhraseFilter: activeKeyPhrase,
                textReviewRanking: currentSortingOption,
            });
            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_MORE_CLICK, {
                hotels: {
                    moreReviewsOffset: data.textReviews.length,
                    activeKeyPhrase,
                },
            });
        }
    }, [getHotelReviews, hotelReviews]);

    const handleSetReviewReaction = useCallback(
        (reactionPayload: IHotelReviewReactionPayload): void => {
            setHotelReviewReaction(reactionPayload);

            debouncedReactionRequest(reactionPayload.reviewId, reactionPayload);
        },
        [debouncedReactionRequest, setHotelReviewReaction],
    );

    const handleReviewsSortingOptionsChange = useCallback(
        (id: TSortId): void => {
            const {isLoadingList: isLoading, activeKeyPhrase} = hotelReviews;

            if (isETextReviewRankingType(id) && !isLoading) {
                sortingHotelReviews({
                    textReviewOffset: 0,
                    textReviewLimit: REVIEWS_LOAD_COUNT,
                    keyPhraseFilter: activeKeyPhrase,
                    textReviewRanking: id,
                });
            }
        },
        [hotelReviews, sortingHotelReviews],
    );

    const renderAnchor = useCallback((): React.ReactNode => {
        return (
            <Anchor
                anchorIds={[HOTEL_REVIEWS_ANCHOR, HOTEL_CREATE_REVIEW_ANCHOR]}
            />
        );
    }, []);

    const renderReviewTotalCount = useCallback((): React.ReactNode => {
        const {
            data: {totalTextReviewCount},
        } = hotelReviews;

        return (
            <Heading
                level={2}
                className={cx(
                    'totalReviewCount',
                    deviceModMobile('totalReviewCount', deviceType),
                )}
                {...prepareQaAttributes('hotelPageReviewsTitle')}
            >
                <Rating className={cx('rating')} size="l" rating={rating} />
                {i18nCommonBlock.totalReviewCount({
                    totalReviewCount: totalTextReviewCount,
                })}
            </Heading>
        );
    }, [deviceType, hotelReviews, rating]);

    const renderTextReview = useCallback(
        (textReview: ITextReview): React.ReactNode => {
            return (
                <HotelReviewWithReaction
                    className={cx('textReview')}
                    key={textReview.id}
                    isAuth={isAuth}
                    review={textReview}
                    onHotelReviewReaction={handleSetReviewReaction}
                />
            );
        },
        [handleSetReviewReaction, isAuth],
    );

    const renderLoader = useCallback((): React.ReactNode => {
        return (
            <Spinner
                className={cx('loader')}
                size={deviceType.isMobile ? 's' : 'xxs'}
                {...prepareQaAttributes('moreReviewsLoader')}
            />
        );
    }, [deviceType]);

    const renderTextReviews = useCallback((): React.ReactNode => {
        const {
            data: {textReviews},
        } = hotelReviews;

        const resultTextReviews = maxReviewRenderCount
            ? textReviews.slice(0, maxReviewRenderCount)
            : textReviews;

        return (
            <div className={cx('reviewsWrap')}>
                {resultTextReviews.map(renderTextReview)}
            </div>
        );
    }, [hotelReviews, maxReviewRenderCount, renderTextReview]);

    const renderSortingOptions = useCallback((): React.ReactNode => {
        const {currentSortingOption, data} = hotelReviews;

        if (!data?.totalTextReviewCount) return null;

        const sortInfo = {
            ...REVIEWS_SORTING_INFO,
            selectedSortId: currentSortingOption,
        };

        // TODO Сделать общий компонент Select + BottomSheet по девайсу
        return (
            <HotelsSortView
                types={sortInfo.availableSortTypeGroups.flatMap(
                    group => group.sortTypes,
                )}
                size="l"
                menuWidth="fixed"
                value={currentSortingOption}
                onChange={handleReviewsSortingOptionsChange}
                showIcon={deviceType.isMobile}
                className={cx('hotelsSort')}
                {...prepareQaAttributes('reviewsSortBar')}
            />
        );
    }, [deviceType, handleReviewsSortingOptionsChange, hotelReviews]);

    const renderKeyPhrase = useCallback(
        (keyPhrase: IKeyPhrase): React.ReactNode => {
            const {activeKeyPhrase} = hotelReviews;

            return (
                <KeyPhraseCheckButton
                    className={cx('keyPhrase')}
                    keyPhrase={keyPhrase}
                    key={keyPhrase.name}
                    checked={keyPhrase.name === activeKeyPhrase}
                    onClick={handleKeyPhraseClick}
                />
            );
        },
        [handleKeyPhraseClick, hotelReviews],
    );

    const renderKeyPhrases = useCallback((): React.ReactNode => {
        const {
            data: {keyPhrases},
        } = hotelReviews;

        if (deviceType.isMobile) {
            return (
                <HorizontalScroller scrollerRef={keyPhrasesRef}>
                    <Flex
                        inline
                        nowrap
                        between={3}
                        className={cx('keyPhrases')}
                        {...prepareQaAttributes('keyPhrases')}
                    >
                        {keyPhrases.map(renderKeyPhrase)}
                    </Flex>
                </HorizontalScroller>
            );
        }

        return (
            <div
                className={cx('keyPhrases')}
                {...prepareQaAttributes('keyPhrases')}
            >
                {keyPhrases.map(renderKeyPhrase)}
            </div>
        );
    }, [deviceType.isMobile, hotelReviews, renderKeyPhrase]);

    const renderMoreReviewsButton = useCallback((): React.ReactNode => {
        const {data, isLoadingList: isLoading} = hotelReviews;

        const moreReviewCount =
            data.totalTextReviewCount - data.textReviews.length;

        if (!moreReviewCount) {
            return null;
        }

        return (
            <div className={cx('moreReviewsButton')}>
                {deviceType.isDesktop && (
                    <>
                        <LinkButton
                            disabled={isLoading}
                            onClick={handleReviewsButtonClick}
                            {...prepareQaAttributes('moreReviewsButton')}
                        >
                            {i18nBlock.moreReviews()}
                        </LinkButton>
                        {isLoading && renderLoader()}
                    </>
                )}
                {deviceType.isMobile && (
                    <>
                        {isLoading ? (
                            <Flex
                                justifyContent="center"
                                alignItems="center"
                                className={cx('loaderContainer')}
                            >
                                {renderLoader()}
                            </Flex>
                        ) : (
                            <Button
                                width="max"
                                size="l"
                                onClick={handleReviewsButtonClick}
                                {...prepareQaAttributes('moreReviewsButton')}
                            >
                                {i18nBlock.moreReviews()}
                            </Button>
                        )}
                    </>
                )}
            </div>
        );
    }, [deviceType, handleReviewsButtonClick, hotelReviews, renderLoader]);

    const renderSkeleton = useCallback((): React.ReactNode => {
        return (
            <TextHotelReviewsSkeleton
                deviceType={deviceType}
                {...prepareQaAttributes('hotelReviewsSkeleton')}
            />
        );
    }, [deviceType]);

    const renderDesktop = useCallback((): React.ReactNode => {
        const {isInitialLoading: isLoadingList, data} = hotelReviews;
        const {userTextReview} = data || {};

        return (
            <>
                {renderAnchor()}
                {renderReviewTotalCount()}
                {renderSortingOptions()}
                {renderKeyPhrases()}
                <Separator />
                <UserReview
                    className={cx('createReview')}
                    userReview={userTextReview}
                />
                <Separator className={cx('createReviewSeparator')} />
                {isLoadingList ? renderSkeleton() : renderTextReviews()}
                {!isLoadingList && renderMoreReviewsButton()}
            </>
        );
    }, [
        hotelReviews,
        renderAnchor,
        renderKeyPhrases,
        renderMoreReviewsButton,
        renderReviewTotalCount,
        renderSkeleton,
        renderSortingOptions,
        renderTextReviews,
    ]);

    const renderMobile = useCallback((): React.ReactNode => {
        const {isInitialLoading: isLoadingList, data} = hotelReviews;
        const {userTextReview} = data || {};

        return (
            <Card className={cx('reviewsCard')}>
                {renderAnchor()}
                {renderReviewTotalCount()}
                {renderSortingOptions()}
                {renderKeyPhrases()}
                {isWeb && (
                    <Card x={4} y={4} background="grey">
                        <UserReview userReview={userTextReview} />
                    </Card>
                )}
                {isLoadingList ? renderSkeleton() : renderTextReviews()}
                {!isLoadingList && renderMoreReviewsButton()}
            </Card>
        );
    }, [
        hotelReviews,
        isWeb,
        renderAnchor,
        renderKeyPhrases,
        renderMoreReviewsButton,
        renderReviewTotalCount,
        renderSkeleton,
        renderSortingOptions,
        renderTextReviews,
    ]);

    return (
        <div
            ref={reviewsRef}
            className={cx('root', deviceMods('root', deviceType), className)}
            {...prepareQaAttributes('hotelReviews')}
        >
            {deviceType.isMobile ? renderMobile() : renderDesktop()}
        </div>
    );
};

export default React.memo(HotelReviews);
