import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {getHotelFeatureRatings} from 'selectors/hotels/hotel/reviewsTab/getHotelFeatureRatings';

import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import cx from './HotelFeatures.scss';

type TRatingMod = 'great' | 'fine' | 'good' | 'none';

const getRatingMod = (percent: number): TRatingMod => {
    if (percent >= 80) {
        return 'great';
    }

    if (percent >= 60) {
        return 'fine';
    }

    if (percent >= 20) {
        return 'good';
    }

    return 'none';
};

interface IWrapperComponentProps extends IWithClassName, IWithQaAttributes {}

export interface IHotelFeaturesProps extends IWithClassName {
    wrapperComponent?: React.ComponentType<IWrapperComponentProps>;
}

const HotelFeatures: React.FC<IHotelFeaturesProps> = ({
    className,
    wrapperComponent: Wrapper = 'div',
}) => {
    const deviceType = useDeviceType();
    const featureRatings = useSelector(getHotelFeatureRatings);

    const canRenderFeatures = featureRatings && featureRatings?.length > 0;

    if (!canRenderFeatures) {
        return null;
    }

    const lines = featureRatings.map(({id, name, positivePercent}) => {
        const fillStyle = {width: `${positivePercent}%`};
        const borderStyle = {
            width: `calc(${positivePercent}% + 2px)`,
        };

        return (
            <div
                key={id}
                className={cx('feature', deviceMods('feature', deviceType))}
            >
                <div>
                    <span className={cx('feature__text')}>{name}</span>
                    <span className={cx('feature__value')}>
                        {`${positivePercent}%`}
                    </span>
                </div>
                <div className={cx('line')}>
                    <div
                        className={cx(
                            'line__left',
                            `line__left_${getRatingMod(positivePercent)}`,
                        )}
                        style={fillStyle}
                    />
                    <div className={cx('line__border')} style={borderStyle} />
                </div>
            </div>
        );
    });

    return (
        <Wrapper
            className={className}
            {...prepareQaAttributes('hotelFeatures')}
        >
            <div className={cx('title', deviceModMobile('title', deviceType))}>
                {i18nBlock.guestsLiked()}
            </div>
            {lines}
        </Wrapper>
    );
};

export default HotelFeatures;
