import {FunctionComponent, ReactNode, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    IKeyPhraseMatch,
    ITextReview,
} from 'types/hotels/hotel/IHotelTextReview';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import TextReviewImages from './components/TextReviewImages/TextReviewImages';
import TruncateText from 'components/TruncateText/TruncateText';

import TextReviewAuthor from '../TextReviewAuthor/TextReviewAuthor';

import cx from './TextHotelReview.scss';

export interface ITextHotelReviewProps extends IWithClassName {
    review: ITextReview;
    topControls?: ReactNode;
    bottomControls?: ReactNode;
}

/**
 * @depricated - заменить на доработанный компонент components/Reviews/Reviews
 * или заменить отдельные отзывы на components/Review/Review */
const TextHotelReview: FunctionComponent<ITextHotelReviewProps> = props => {
    const {
        className,
        topControls,
        bottomControls,
        review: {id, author, text, rating, updatedAt, images, keyPhraseMatch},
    } = props;

    const deviceType = useDeviceType();

    const textNode = useMemo(
        () => renderTextNode(text, keyPhraseMatch),
        [text, keyPhraseMatch],
    );

    const textContentNode = useMemo(
        () =>
            deviceType.isMobile ? <TruncateText text={textNode} /> : textNode,
        [deviceType, textNode],
    );

    return (
        <section
            className={cx(deviceMods('root', deviceType), className)}
            {...prepareQaAttributes({
                key: id,
                current: 'hotelPageReview',
            })}
        >
            {topControls}
            {author && (
                <TextReviewAuthor
                    className={cx('author')}
                    author={author}
                    rating={rating}
                    date={updatedAt}
                />
            )}
            <div className={cx('text')} {...prepareQaAttributes('text')}>
                {textContentNode}
            </div>
            {images?.length ? (
                <TextReviewImages images={images} className={cx('images')} />
            ) : null}
            {bottomControls}
        </section>
    );
};

function renderTextNode(
    text: string,
    keyPhraseMatch?: IKeyPhraseMatch,
): ReactNode {
    if (!keyPhraseMatch?.fragments.length) {
        return text;
    }

    const init: {
        nodes: ReactNode[];
        prevFragmentEnd: number;
    } = {
        nodes: [],
        prevFragmentEnd: 0,
    };

    const reduced = keyPhraseMatch.fragments
        .sort((match1, match2): number => match1.position - match2.position)
        .reduce((result, fragment) => {
            const beforeText = text.slice(
                result.prevFragmentEnd,
                fragment.position,
            );
            const fragmentText = text.substr(fragment.position, fragment.size);

            result.nodes.push(beforeText);
            result.nodes.push(
                <span className={cx('keyPhrase')}>{fragmentText}</span>,
            );

            result.prevFragmentEnd = fragment.position + fragment.size;

            return result;
        }, init);

    reduced.nodes.push(text.slice(reduced.prevFragmentEnd));

    return reduced.nodes;
}

export default TextHotelReview;
