import React, {useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Flex from 'components/Flex/Flex';
import TravelImage from 'components/TravelImage/TravelImage';
import GalleryRow from 'projects/hotels/pages/HotelPage/components/GalleryRow/GalleryRow';

import cx from './TextReviewGallery.scss';

const MAX_DESKTOP_IMAGES = 8;

interface ITextReviewGalleryProps extends IWithClassName {
    imageUrls: string[];
    onImageClick: (index: number) => void;
}

const TextReviewGallery: React.FC<ITextReviewGalleryProps> = ({
    className,
    imageUrls,
    onImageClick,
}) => {
    const {isDesktop} = useDeviceType();

    const desktopImagesUrls = useMemo(
        () => imageUrls.slice(0, MAX_DESKTOP_IMAGES),
        [imageUrls],
    );

    const handleImageClick = useCallback(
        (e, id) => onImageClick(id),
        [onImageClick],
    );

    if (isDesktop) {
        return (
            <GalleryRow
                isLastClickable={imageUrls.length > MAX_DESKTOP_IMAGES}
                imageSources={desktopImagesUrls}
                totalImageCount={imageUrls.length}
                onImageClick={onImageClick}
                onMoreImagesClick={onImageClick}
                className={cx('root_desktop', className)}
                imageWrapperClassName={cx('image')}
            />
        );
    }

    return (
        <HorizontalScroller className={className} offset={3}>
            <Flex inline nowrap between={1}>
                {imageUrls.map((url, index) => (
                    <TravelImage
                        key={url}
                        src={url}
                        id={index}
                        className={cx('image')}
                        onClick={handleImageClick}
                        hasFitCover
                    />
                ))}
            </Flex>
        </HorizontalScroller>
    );
};

export default TextReviewGallery;
