import {MouseEventHandler, FunctionComponent, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ITextReviewAuthor} from 'types/hotels/hotel/IHotelTextReview';
import {EHotelImageYapicSize} from 'types/hotels/common/EHotelImageYapicSize';

import {humanFormatDate} from 'utilities/dateUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {parseYapicImageUrlTemplate} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import getAvatarUrl from 'utilities/avatar/getAvatarUrl';
import {DEFAULT_AVATAR_ID} from 'utilities/avatar/getAvatarId';

import Link from 'components/Link/Link';
import Stars from 'components/Stars/Stars';
import Avatar from 'components/Avatar/Avatar';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './TextReviewAuthor.scss';

/* Constants */
const MAX_REVIEW_STARS = 5;

/* Component Types */
interface ITextReviewAuthorProps extends IWithClassName {
    author: ITextReviewAuthor;
    rating: number;
    date: string;
}

const TextReviewAuthor: FunctionComponent<ITextReviewAuthorProps> = props => {
    const {
        className,
        rating,
        date,
        author: {avatarUrl, name, profileUrl},
    } = props;

    const userInfo = useUserInfo();
    const {isWeb} = usePlatform();

    const defaultAvatarUrl = getAvatarUrl(
        userInfo.avatarPath,
        DEFAULT_AVATAR_ID,
        {retina: true},
    );

    const avatarSrc = avatarUrl
        ? parseYapicImageUrlTemplate(
              avatarUrl,
              EHotelImageYapicSize.RETINA_MIDDLE,
          )
        : defaultAvatarUrl;

    const handleLinkClick: MouseEventHandler = useCallback(
        e => {
            // Для вставки в мобильные приложения блокируем переход
            if (!isWeb) {
                e.preventDefault();
            }
        },
        [isWeb],
    );

    return (
        <section className={cx(className, 'author')}>
            <Link
                className={cx('avatar')}
                target="_blank"
                rel="noopener noreferrer"
                url={profileUrl}
                onClick={handleLinkClick}
            >
                <Avatar src={avatarSrc} size="m" />
            </Link>
            <div className={cx('authorInfo')}>
                <div className={cx('about')}>
                    <Link
                        className={cx('name')}
                        theme={isWeb ? undefined : 'black'}
                        target="_blank"
                        rel="noopener noreferrer"
                        url={profileUrl}
                        onClick={handleLinkClick}
                        {...prepareQaAttributes('authorName')}
                    >
                        {name}
                    </Link>
                    <span
                        className={cx('date')}
                        {...prepareQaAttributes('hotelReviewDate')}
                    >
                        {humanFormatDate(date)}
                    </span>
                </div>
                <Stars
                    className={cx('stars')}
                    starCount={rating}
                    totalStarCount={MAX_REVIEW_STARS}
                />
            </div>
        </section>
    );
};

export default TextReviewAuthor;
