import React, {useCallback, useState, useEffect} from 'react';
import {useDispatch} from 'react-redux';

import {
    HOTEL_CREATE_REVIEW_ANCHOR,
    HOTEL_REVIEWS_ANCHOR,
} from 'constants/urls/anchors';

import {IWithClassName} from 'types/withClassName';
import {ITextReview} from 'types/hotels/hotel/IHotelTextReview';
import {IEditHotelReviewRequest} from 'server/api/HotelSearchAPI/types/IEditHotelReview';
import {IAddHotelReviewRequest} from 'server/api/HotelSearchAPI/types/IAddHotelReview';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {resetAddHotelReviewAction} from 'reducers/hotels/hotelPage/reviews/add/actions';
import {resetEditHotelReviewAction} from 'reducers/hotels/hotelPage/reviews/edit/actions';

import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useTimeout} from 'utilities/hooks/useTimeout';
import {
    clearSelectedStars,
    restoreSelectedStars,
    storeSelectedStars,
} from './utilities/storedStars';
import {reachGoal} from 'utilities/metrika';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import getUrlWithUpdatedHash from 'utilities/url/getUrlWithUpdatedHash';
import {useUserInfo} from 'utilities/hooks/useUserInfo';

import CreateReviewDialog from './components/CreateReviewDialog/CreateReviewDialog';
import CreateReviewSuccessDialog from './components/CreateReviewSuccessDialog/CreateReviewSuccessDialog';
import CreateReviewTrigger from './components/CreateReviewTrigger/CreateReviewTrigger';
import EditReviewDialog from './components/EditReviewDialog/EditReviewDialog';
import UserTextReview from './components/UserTextReview/UserTextReview';
import DeleteReviewDialog from './components/DeleteReviewDialog/DeleteReviewDialog';
import ErrorDialog from './components/ErrorDialog/ErrorDialog';
import ImageUploaderContextProvider from './components/ImageUploader/ImageUploaderContext';

import cx from './UserReview.scss';

const MODAL_TIMEOUT = 300; // ждем пока закроется предыдущий модал

interface IUserReviewProps extends IWithClassName {
    userReview?: ITextReview;
}

const UserReview: React.FC<IUserReviewProps> = ({userReview, className}) => {
    const reviewStars = restoreSelectedStars();
    const [selectedStars, setSelectedStars] = useState(reviewStars || 0);
    const [retryValues, setRetryValues] = useState<
        IAddHotelReviewRequest | IEditHotelReviewRequest
    >({});
    const deviceType = useDeviceType();
    const dispatch = useDispatch();

    const {
        value: isCreateDialogVisible,
        setTrue: showCreateDialog,
        setFalse: hideCreateDialog,
    } = useBoolean(false);
    const {
        value: isSuccessDialogVisible,
        setTrue: showSuccessDialog,
        setFalse: hideSuccessDialog,
    } = useBoolean(false);
    const {
        value: isEditDialogVisible,
        setTrue: showEditDialog,
        setFalse: hideEditDialog,
    } = useBoolean(false);
    const {
        value: isDeleteDialogVisible,
        setTrue: showDeleteDialog,
        setFalse: hideDeleteDialog,
    } = useBoolean(false);
    const {
        value: isErrorDialogVisible,
        setTrue: showErrorDialog,
        setFalse: hideErrorDialog,
    } = useBoolean(false);

    const showSuccessDialogWithTimeout = useTimeout(
        showSuccessDialog,
        MODAL_TIMEOUT,
    );
    const showErrorDialogWithTimeout = useTimeout(
        showErrorDialog,
        MODAL_TIMEOUT,
    );

    const userInfo = useUserInfo();
    const isAuth = isAuthUser(userInfo);

    const handleRatingStartClick = useCallback(
        (star: number) => {
            if (isAuth) {
                setSelectedStars(star);
                showCreateDialog();
            } else {
                storeSelectedStars(star);

                const {passportPath} = userInfo;

                window.open(
                    getPassportUrl({
                        mode: EPassportMode.ADD_USER,
                        passportHost: passportPath,
                        retpath: getUrlWithUpdatedHash(
                            location.href,
                            HOTEL_REVIEWS_ANCHOR,
                        ),
                    }),
                    '_self',
                );
            }

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_STARS_CLICK, {
                hotels: {
                    reviewStars: star,
                },
            });
        },
        [setSelectedStars, showCreateDialog, userInfo, isAuth],
    );

    const handleSubmitReviewSuccess = useCallback(() => {
        hideCreateDialog();
        hideEditDialog();
        showSuccessDialogWithTimeout();
        reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_THANKYOU_TOOLTIP);
    }, [hideCreateDialog, showSuccessDialogWithTimeout, hideEditDialog]);

    const handleCreateError = useCallback(
        values => {
            hideCreateDialog();
            setRetryValues(values);
            showErrorDialogWithTimeout();

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_SEND_ERROR);
        },
        [showErrorDialogWithTimeout, setRetryValues, hideCreateDialog],
    );
    const handleEditError = useCallback(
        values => {
            hideEditDialog();
            setRetryValues(values);
            showErrorDialogWithTimeout();

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_SEND_ERROR);
        },
        [setRetryValues, hideEditDialog, showErrorDialogWithTimeout],
    );
    const handleDeleteClick = useCallback(() => {
        showDeleteDialog();
        reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_DELETE_MODAL_SHOW);
    }, [showDeleteDialog]);
    const handleEditClick = useCallback(() => {
        showEditDialog();
        reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_EDIT_CLICK);
    }, [showEditDialog]);
    const handleErrorDialogClose = useCallback(() => {
        hideErrorDialog();
        dispatch(resetAddHotelReviewAction());
        dispatch(resetEditHotelReviewAction());
    }, [hideErrorDialog, dispatch]);

    useEffect(() => {
        if (
            isAuth &&
            (Number.isInteger(reviewStars) ||
                location.hash === `#${HOTEL_CREATE_REVIEW_ANCHOR}`)
        ) {
            clearSelectedStars();

            if (!userReview) {
                showCreateDialog();
            }
        }
    }, []);

    return (
        <ImageUploaderContextProvider>
            {userReview ? (
                <UserTextReview
                    className={cx(deviceMods('review', deviceType))}
                    review={userReview}
                    onEditClick={handleEditClick}
                    onDeleteClick={handleDeleteClick}
                />
            ) : (
                <CreateReviewTrigger
                    className={className}
                    onStarClick={handleRatingStartClick}
                />
            )}
            <CreateReviewDialog
                isVisible={isCreateDialogVisible}
                stars={selectedStars}
                onSuccess={handleSubmitReviewSuccess}
                onClose={hideCreateDialog}
                onError={handleCreateError}
            />
            <CreateReviewSuccessDialog
                isVisible={isSuccessDialogVisible}
                onClose={hideSuccessDialog}
            />
            {userReview && (
                <EditReviewDialog
                    isVisible={isEditDialogVisible}
                    userReview={userReview}
                    onSuccess={handleSubmitReviewSuccess}
                    onClose={hideEditDialog}
                    onError={handleEditError}
                />
            )}
            <DeleteReviewDialog
                isVisible={isDeleteDialogVisible}
                userReview={userReview}
                onClose={hideDeleteDialog}
            />
            <ErrorDialog
                isVisible={isErrorDialogVisible}
                onClose={handleErrorDialogClose}
                retryValues={retryValues}
            />
        </ImageUploaderContextProvider>
    );
};

export default UserReview;
