import React, {useCallback, useEffect, useMemo, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {ITextReview} from 'types/hotels/hotel/IHotelTextReview';
import {IEditHotelReviewRequest} from 'server/api/HotelSearchAPI/types/IEditHotelReview';
import {TImagesSelectorImage} from 'components/ImagesSelector/types/IImagesSelectorImage';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';

import {editHotelReviewActions} from 'reducers/hotels/hotelPage/reviews/edit/actions';

import editHotelReview from 'selectors/hotels/hotel/reviewsTab/editHotelReview';

import {parseAltaiImageUrlTemplate} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

import ReviewDialog, {IReviewDialogValues} from '../ReviewDialog/ReviewDialog';

interface IEditReviewDialogProps {
    isVisible: boolean;
    userReview: ITextReview;
    onClose: () => void;
    onSuccess: () => void;
    onError: (payload: IEditHotelReviewRequest) => void;
}

const EditReviewDialog: React.FC<IEditReviewDialogProps> = ({
    isVisible,
    userReview,
    onClose,
    onSuccess,
    onError,
}) => {
    const {isSubmitting, isSuccess, hasError} = useSelector(editHotelReview);
    const [submittedValues, setSubmittedValues] =
        useState<IEditHotelReviewRequest>({
            reviewId: userReview.id,
        });
    const dispatch = useDispatch();
    const editReview = useCallback(
        ({reviewImages, ...restValue}: IReviewDialogValues) => {
            const payload = {
                reviewId: userReview.id,
                reviewImageIds: reviewImages.map(({id}) => id),
                ...restValue,
            };

            setSubmittedValues(payload);
            dispatch(editHotelReviewActions.request(payload));
        },
        [dispatch, userReview],
    );
    const initialValues = useMemo(
        () => ({
            reviewRating: userReview.rating,
            reviewText: userReview.text,
            reviewImages:
                userReview.images?.map(({id, urlTemplate, sizes}) => ({
                    id,
                    src: parseAltaiImageUrlTemplate(
                        urlTemplate,
                        EHotelImageAltayStandardSize.MINI,
                        sizes,
                    ),
                })) || ([] as TImagesSelectorImage[]),
        }),
        [userReview, isVisible],
    );

    useEffect(() => {
        if (isSuccess) {
            onSuccess();
        }
    }, [isSuccess, onSuccess]);

    useEffect(() => {
        if (hasError && isVisible) {
            onClose();
            onError(submittedValues);
        }
    }, [hasError, isVisible, submittedValues, onClose, onError]);

    return (
        <ReviewDialog
            isVisible={isVisible}
            isSubmitting={isSubmitting}
            onClose={onClose}
            onSubmit={editReview}
            initialValues={initialValues}
            prohibitEmptyComment={true}
        />
    );
};

export default EditReviewDialog;
