import React, {ReactNode, useCallback, useState} from 'react';
import {useSelector} from 'react-redux';

import {TImagesSelectorImage} from 'components/ImagesSelector/types/IImagesSelectorImage';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import hotelPageSelector from 'selectors/hotels/hotel/hotelPageSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {validateReviewText} from './utilities/validateReviewText';
import {deviceMods} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useTimeout} from 'utilities/hooks/useTimeout';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import Modal from 'components/Modal/Modal';
import TextArea from 'components/TextArea/TextArea';
import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import RatingStars from 'components/RatingStars/RatingStars';
import Button from 'components/Button/Button';
import Form from 'components/Form/Form';
import FormField from 'components/FormField/FormField';
import ImagesSelectorAdaptor from './components/ImagesSelector/ImagesSelectorAdaptor';
import Snackbar from 'components/Snackbar/Snackbar';
import Flex from 'components/Flex/Flex';

import {useImageUploader} from '../ImageUploader/useImageUploader';

import cx from './ReviewDialog.scss';

enum EFieldName {
    REVIEW_RATING = 'reviewRating',
    REVIEW_TEXT = 'reviewText',
    REVIEW_IMAGES = 'reviewImages',
}

const UPLOADING_IMAGES_MESSAGE_HIDE_TIMEOUT = 3000;

export interface IReviewDialogValues {
    [EFieldName.REVIEW_RATING]: number;
    [EFieldName.REVIEW_TEXT]: string;
    [EFieldName.REVIEW_IMAGES]: TImagesSelectorImage[];
}
interface IReviewDialogProps {
    isVisible: boolean;
    isSubmitting: boolean;
    initialValues: IReviewDialogValues;
    onClose: () => void;
    onSubmit: (values: IReviewDialogValues) => void;
    prohibitEmptyComment?: boolean;
}

const ReviewDialog: React.FC<IReviewDialogProps> = ({
    isVisible,
    isSubmitting,
    initialValues,
    onClose,
    onSubmit,
    prohibitEmptyComment,
}) => {
    const {
        hotelInfo: {data: hotelData},
    } = useSelector(hotelPageSelector);
    const hotelSlug = hotelData?.hotel?.hotelSlug;
    const hotelName = hotelData?.hotel?.name;
    const deviceType = useDeviceType();
    const {deleteUnsavedImages, deleteServerImages, hasUploadingImages} =
        useImageUploader(hotelSlug);
    const {
        value: isUploadingImagesMessageVisible,
        setTrue: showUploadingImagesMessage,
        setFalse: hideUploadingImagesMessage,
    } = useBoolean(false);
    const [bigFilesCount, setBigFilesCount] = useState(0);
    const {
        value: isBigFileSizeErrorVisible,
        setTrue: showBigFileSizeError,
        setFalse: hideBigFileSizeError,
    } = useBoolean(false);
    const hideUploadingImagesMessageWithTimeout = useTimeout(
        hideUploadingImagesMessage,
        UPLOADING_IMAGES_MESSAGE_HIDE_TIMEOUT,
    );
    const hideBigFileSizeErrorWithTimeout = useTimeout(
        hideBigFileSizeError,
        UPLOADING_IMAGES_MESSAGE_HIDE_TIMEOUT,
    );

    const handleBigFileSizeError = useCallback(
        count => {
            setBigFilesCount(count);
            showBigFileSizeError();
            hideBigFileSizeErrorWithTimeout();
            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_LARGEPHOTO_TOOLTIP);
        },
        [
            setBigFilesCount,
            showBigFileSizeError,
            hideBigFileSizeErrorWithTimeout,
        ],
    );

    const handleClose = useCallback(() => {
        deleteUnsavedImages();
        onClose();
    }, [deleteUnsavedImages, onClose]);

    const handleSubmit = useCallback(
        (values: IReviewDialogValues) => {
            if (hasUploadingImages()) {
                showUploadingImagesMessage();
                hideUploadingImagesMessageWithTimeout();

                return;
            }

            deleteServerImages();

            onSubmit(values);

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_SEND_CLICK);
        },
        [
            onSubmit,
            showUploadingImagesMessage,
            hideUploadingImagesMessageWithTimeout,
            deleteServerImages,
            hasUploadingImages,
        ],
    );

    return (
        <>
            <Modal
                isVisible={isVisible}
                onClose={handleClose}
                className={cx(deviceMods('modal', deviceType))}
                dialogClassName={cx('modalDialog')}
                containerClassName={cx('modalContainer')}
                fullScreen={deviceType.isMobile}
            >
                <Form onSubmit={handleSubmit} initialValues={initialValues}>
                    {({handleSubmit}): ReactNode => (
                        <form onSubmit={handleSubmit} className={cx('form')}>
                            <Flex
                                flexDirection="column"
                                className={cx('titleWrap')}
                            >
                                <Heading level="3" weight="bold">
                                    {i18nBlock.rateAndLeaveReview()}
                                </Heading>
                                <Text
                                    size="s"
                                    color="secondary"
                                    tag="div"
                                    className={cx('subtitle')}
                                >
                                    {hotelName}
                                </Text>
                            </Flex>
                            <Flex
                                flexDirection="column"
                                className={cx('scrollableWrap')}
                            >
                                <FormField
                                    name={EFieldName.REVIEW_RATING}
                                    deviceType={deviceType}
                                    control={({input}): ReactNode => (
                                        <RatingStars
                                            className={cx('stars')}
                                            selected={input.value}
                                            onClick={(value: number): void =>
                                                input.onChange({
                                                    target: {value},
                                                })
                                            }
                                        />
                                    )}
                                />
                                <Text className={cx('commentLabel')} tag="div">
                                    {i18nBlock.comment()}
                                </Text>
                                <FormField
                                    name={EFieldName.REVIEW_TEXT}
                                    deviceType={deviceType}
                                    validate={validateReviewText({
                                        prohibitEmpty: prohibitEmptyComment,
                                    })}
                                    control={(
                                        {input},
                                        {error, controlRef},
                                    ): ReactNode => (
                                        <TextArea
                                            className={cx('comment')}
                                            rows={deviceType.isDesktop ? 4 : 7}
                                            value={input.value}
                                            onChange={input.onChange}
                                            state={error ? 'error' : undefined}
                                            placeholder={i18nBlock.commentFieldPlaceholder()}
                                            wrapRef={controlRef}
                                        />
                                    )}
                                />

                                <Text className={cx('commentLabel')} tag="div">
                                    {i18nBlock.completeReviewWithPhotos()}
                                </Text>
                                <FormField
                                    name={EFieldName.REVIEW_IMAGES}
                                    deviceType={deviceType}
                                    control={({input}): ReactNode => (
                                        <ImagesSelectorAdaptor
                                            className={cx('images')}
                                            images={input.value}
                                            hotelSlug={hotelSlug}
                                            onBigFileSizeError={
                                                handleBigFileSizeError
                                            }
                                            onChange={(images): void => {
                                                input.onChange({
                                                    target: {
                                                        value: images,
                                                    },
                                                });
                                            }}
                                        />
                                    )}
                                />
                            </Flex>
                            <div className={cx('buttonWrap')}>
                                <Button
                                    className={cx('submit')}
                                    theme="primary"
                                    width="max"
                                    size="l"
                                    type="submit"
                                    disabled={isSubmitting}
                                >
                                    {i18nBlock.submit()}
                                </Button>
                            </div>
                        </form>
                    )}
                </Form>
            </Modal>
            <Snackbar isVisible={isUploadingImagesMessageVisible}>
                {i18nBlock.pleaseWaitPhotoUpload()}
            </Snackbar>
            <Snackbar isVisible={isBigFileSizeErrorVisible}>
                {i18nBlock.bigFileSizeError({bigFilesCount})}
            </Snackbar>
        </>
    );
};

export default ReviewDialog;
