import React, {useCallback, useEffect, useRef} from 'react';

import {PHOTO_UPLOAD_LIMIT_IN_BYTES} from 'server/constants/hotels';

import {IWithClassName} from 'types/withClassName';
import {
    INewImagesSelectorImage,
    TImagesSelectorImage,
} from 'components/ImagesSelector/types/IImagesSelectorImage';
import EAsyncStatus from 'types/common/EAsyncStatus';

import ImagesSelector from 'components/ImagesSelector/ImagesSelector';

import {useImageUploader} from '../../../ImageUploader/useImageUploader';

interface IImagesSelectorProps extends IWithClassName {
    hotelSlug?: string;
    images: TImagesSelectorImage[];
    onChange: (images: TImagesSelectorImage[]) => void;
    onBigFileSizeError: (bigFilesCount: number) => void;
}

const ImagesSelectorAdaptor: React.FC<IImagesSelectorProps> = ({
    hotelSlug,
    images,
    onChange,
    onBigFileSizeError,
    ...restProps
}) => {
    const {
        images: uploadingImages,
        uploadImages,
        deleteImage,
    } = useImageUploader(hotelSlug);
    const uploadingImagesLength = useRef<number>(0);

    useEffect(() => {
        const loadingImagesCount = uploadingImages.filter(
            i => i.status === EAsyncStatus.LOADING,
        ).length;

        if (uploadingImagesLength.current !== loadingImagesCount) {
            uploadingImagesLength.current = loadingImagesCount;
            onChange(images);
        }
    }, [onChange, images, uploadingImages]);

    const handleImagesSelected = useCallback(
        (newImages: INewImagesSelectorImage[]) => {
            const suitableImages = newImages.filter(
                i => i.file.size <= PHOTO_UPLOAD_LIMIT_IN_BYTES,
            );

            if (suitableImages.length > 0) {
                onChange([...images, ...suitableImages]);
                uploadImages(suitableImages);
            }

            if (suitableImages.length !== newImages.length) {
                onBigFileSizeError(newImages.length - suitableImages.length);
            }
        },
        [images, onChange, uploadImages, onBigFileSizeError],
    );

    const handleDeleteClick = useCallback(
        imageToDelete => {
            onChange(images.filter(i => i !== imageToDelete));
            deleteImage(imageToDelete);
        },
        [images, onChange, deleteImage],
    );

    return (
        <ImagesSelector
            images={images}
            onImagesSelected={handleImagesSelected}
            onDeleteClick={handleDeleteClick}
            {...restProps}
        />
    );
};

export default ImagesSelectorAdaptor;
