import {FunctionComponent} from 'react';

import {ITextReview} from 'types/hotels/hotel/IHotelTextReview';

import {IAuthUserInfo} from 'reducers/common/commonReducerTypes';

import getAvatarUrl from 'utilities/avatar/getAvatarUrl';
import {DEFAULT_AVATAR_ID} from 'utilities/avatar/getAvatarId';
import {useUserInfo} from 'utilities/hooks/useUserInfo';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

/* Components */
import LinkButton from 'components/LinkButton/LinkButton';
import EditIcon from 'icons/16/Edit';
import DeleteIcon from 'icons/16/Delete';
import Flex from 'components/Flex/Flex';
import UserTextReviewStatus from './components/UserTextReviewStatus/UserTextReviewStatus';

import TextHotelReview, {
    ITextHotelReviewProps,
} from '../../../TextHotelReview/TextHotelReview';

import cx from './UserTextReview.scss';

interface IUserTextReviewProps extends Omit<ITextHotelReviewProps, 'controls'> {
    onEditClick: () => void;
    onDeleteClick: () => void;
}

const UserTextReview: FunctionComponent<IUserTextReviewProps> = props => {
    const {onEditClick, onDeleteClick, review, ...restProps} = props;

    const userInfo = useUserInfo() as IAuthUserInfo;

    const fixedReview = fixReview(props.review, userInfo);

    return (
        <TextHotelReview
            review={fixedReview}
            topControls={
                <UserTextReviewStatus
                    className={cx('status')}
                    review={review}
                    userInfo={userInfo}
                />
            }
            bottomControls={
                <Flex
                    flexDirection="row"
                    alignItems="center"
                    between={3}
                    inline
                >
                    <LinkButton onClick={onEditClick} className={cx('button')}>
                        <EditIcon className={cx('icon')} />
                        {fixedReview.text
                            ? i18nBlock.edit()
                            : i18nBlock.writeReview()}
                    </LinkButton>
                    <LinkButton
                        onClick={onDeleteClick}
                        className={cx('button')}
                    >
                        <DeleteIcon className={cx('icon')} />
                        {i18nBlock._delete()}
                    </LinkButton>
                </Flex>
            }
            {...restProps}
        />
    );
};

/**
 * author.name и author.avatarUrl приходят пустыми
 * можно удалить когда баг пофиксят
 */
function fixReview(review: ITextReview, userInfo: IAuthUserInfo): ITextReview {
    if (!review.author?.avatarUrl) {
        const author = review.author || {
            name: userInfo.displayName,
            profileUrl: '',
            level: '',
        };

        return {
            ...review,
            author: {
                ...author,
                avatarUrl: getAvatarUrl(userInfo.avatarPath, DEFAULT_AVATAR_ID),
            },
        };
    }

    return review;
}

export default UserTextReview;
