import React, {useCallback, useRef} from 'react';

import {
    ETextReviewModerationStatus,
    ITextReview,
} from 'types/hotels/hotel/IHotelTextReview';
import {IWithClassName} from 'types/withClassName';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {IAuthUserInfo} from 'reducers/common/commonReducerTypes';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotels-HotelReviews';
import * as i18nCommonBlock from 'i18n/common';

import Label, {ELabelThemeType} from 'components/Label/Label';
import Flex from 'components/Flex/Flex';
import InfoIcon from 'icons/16/Info';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import LinkButton from 'components/LinkButton/LinkButton';
import InfoS from 'icons/illustrations/Info/InfoS';

import cx from './UserTextReviewStatus.scss';

interface IUserTextReviewStatusProps extends IWithClassName {
    review: ITextReview;
    userInfo: IAuthUserInfo;
}

const UserTextReviewStatus: React.FC<IUserTextReviewStatusProps> = ({
    review,
    userInfo,
    className,
}) => {
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;
    const {
        value: isDeclineMessageVisible,
        setTrue: showDeclineMessage,
        setFalse: hideDeclineMessage,
    } = useBoolean(false);
    const anchorRef = useRef(null);
    const handleDeclineQuestionClick = useCallback(() => {
        showDeclineMessage();
        reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_REJECT_MODAL_SHOW);
    }, [showDeclineMessage]);

    if (!review.moderation) {
        return null;
    }

    const {status} = review.moderation;

    if (status === ETextReviewModerationStatus.PENDING) {
        return (
            <Label className={className} theme={ELabelThemeType.NEUTRAL}>
                {i18nBlock.reviewStatusPendingLabel()}
            </Label>
        );
    }

    if (status === ETextReviewModerationStatus.DECLINED) {
        return (
            <>
                <Flex
                    flexDirection="row"
                    alignItems="center"
                    justifyContent={isMobile ? 'space-between' : 'flex-start'}
                    between={2}
                    inline
                    className={className}
                >
                    <Label className={cx('decline')}>
                        {i18nBlock.reviewStatusDeclinedLabel()}
                    </Label>
                    <span
                        className={cx('declineIconWrap')}
                        ref={anchorRef}
                        onClick={handleDeclineQuestionClick}
                    >
                        <InfoIcon className={cx('declineIcon')} />
                    </span>
                </Flex>
                <MessageBoxPopupOrBottomSheet
                    isVisible={isDeclineMessageVisible}
                    onClose={hideDeclineMessage}
                    messageBoxProps={{
                        anchorRef: anchorRef,
                        theme: EMessageBoxPopupTheme.DEFAULT,
                        direction: EPopupDirection.BOTTOM_LEFT,
                        tailOffset: 21,
                        secondaryOffset: -10,
                        className: cx('messageBox'),
                        closeOnScroll: true,
                    }}
                    bottomSheetProps={{
                        hasCloseButton: true,
                    }}
                >
                    <Flex flexDirection="column" alignItems="center">
                        {isMobile && <InfoS className={cx('image')} />}
                        <div className={cx(deviceMods('message', deviceType))}>
                            {i18nBlock.declineReason()}
                        </div>
                        {isMobile ? (
                            <ButtonLink
                                theme="primary"
                                width="max"
                                size="l"
                                url={userInfo.passportPath}
                                target="_blank"
                            >
                                {i18nBlock.show()}
                            </ButtonLink>
                        ) : (
                            <Flex
                                flexDirection="row"
                                justifyContent="flex-end"
                                className={cx('buttons')}
                            >
                                <LinkButton
                                    theme="ghost"
                                    className={cx('closeButton')}
                                    onClick={hideDeclineMessage}
                                >
                                    {i18nCommonBlock.close()}
                                </LinkButton>
                                <ButtonLink
                                    theme="primary"
                                    size="m-inset"
                                    url={userInfo.passportPath}
                                    target="_blank"
                                    className={cx('show')}
                                >
                                    {i18nBlock.show()}
                                </ButtonLink>
                            </Flex>
                        )}
                    </Flex>
                </MessageBoxPopupOrBottomSheet>
            </>
        );
    }

    return (
        <Label className={className} theme={ELabelThemeType.NEUTRAL}>
            {i18nBlock.reviewStatusAcceptedLabel()}
        </Label>
    );
};

export default UserTextReviewStatus;
