import React, {CSSProperties, RefObject, useCallback, useMemo} from 'react';
import {ParsedQuery} from 'query-string';

import {MIN_PERCENT_TO_SHOW} from 'projects/hotels/constants/discount';

/* Types */
import {
    IHotelOffer,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import IPrice from 'types/common/price/IPrice';
import {IHotel} from 'types/hotels/hotel/IHotel';
import {IWithClassName} from 'types/withClassName';
import {EPortalSeedReferer} from 'types/hotels/common/IAttribution';
import {IHotelPageQueryParams} from 'types/hotels/common/IQueryParams';

import {CHAR_MIDDLE_DOT, CHAR_NBSP} from 'utilities/strings/charCodes';
import {parseImageUrlTemplate} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    hotelsURLs,
    getSearchParamsForBaseHotelUrl,
} from 'projects/hotels/utilities/urls';

import * as i18nShortViewBlock from 'i18n/hotels-HotelShortView';
import * as i18nBlock from 'i18n/hotels-Common';

import Price from 'components/Price/Price';
import TravelImage from 'components/TravelImage/TravelImage';
import Link from 'components/Link/Link';
import Rating from 'components/Rating/Rating';
import HotelImageStub from 'projects/hotels/components/HotelImageStub/HotelImageStub';
import Flex from 'components/Flex/Flex';
import DiscountInfo from 'projects/hotels/components/HotelCard/components/DiscountInfo/DiscountInfo';
import HotelStars from 'components/HotelStars/HotelStars';

import cx from './HotelShortView.scss';

/* Types */
export interface IHotelShortViewProps
    extends IWithClassName,
        IWithQaAttributes {
    titleClassName?: string;
    style?: CSSProperties;
    searchParams?: Partial<IRequiredOfferParams>;
    queryByLocation?: ParsedQuery;
    seed?: EPortalSeedReferer;
    canRenderOfferPrice?: boolean;
    canRenderAddress?: boolean;
    hideImage?: boolean;
    rootRef?: RefObject<HTMLDivElement>;
    onClickHotelName?: () => void;
    imageSizeName?: string;
    hotel: IHotel;
    offers?: IHotelOffer[];
    minPrice?: IPrice;
    queryParams?: IHotelPageQueryParams;
    imageClassName?: string;
    hotelsPercentDiscount?: boolean;
}

const BUILDING_ILLUSTRATION_PROPS = {height: 20, width: 22};
const imageStub = (
    <HotelImageStub illustrationProps={BUILDING_ILLUSTRATION_PROPS} />
);

const HotelShortView = (props: IHotelShortViewProps): React.ReactElement => {
    const {
        seed,
        className,
        titleClassName,
        canRenderOfferPrice,
        canRenderAddress,
        hideImage,
        style,
        rootRef,
        queryByLocation,
        searchParams,
        onClickHotelName,
        imageSizeName,
        hotel,
        offers,
        minPrice,
        queryParams,
        imageClassName,
        hotelsPercentDiscount,
    } = props;

    const isWithoutDatesPage = !searchParams;

    const {
        name,
        address,
        totalTextReviewCount,
        stars,
        rating,
        images,
        category: {name: categoryName},
    } = hotel;

    const [image] = images;
    const imageSrc =
        image &&
        parseImageUrlTemplate(image.urlTemplate, imageSizeName, image.sizes);

    const params = getSearchParamsForBaseHotelUrl({
        seed,
        hotel,
        offers,
        offerRequestParams: searchParams,
    });

    const handleClickHotelName = useCallback(() => {
        if (onClickHotelName) {
            onClickHotelName();
        }
    }, [onClickHotelName]);
    const hotelLandingUrl = hotelsURLs.getHotelUrl(
        params,
        queryByLocation,
        queryParams,
    );

    const priceBlock = useMemo(() => {
        if (minPrice) {
            return minPrice ? (
                <Flex inline between={2} above={1}>
                    <Price
                        className={cx('firstOfferPrice')}
                        {...minPrice}
                        isFrom
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'minPrice',
                        })}
                    />
                </Flex>
            ) : (
                <span
                    className={cx(
                        'firstOfferPrice',
                        'firstOfferPrice_noOffers',
                    )}
                >
                    {i18nShortViewBlock.noOffersNew()}
                </span>
            );
        }

        const firstOffer = offers?.[0];
        const firstOfferPrice = firstOffer?.price;
        const firstOfferDiscount = firstOffer?.discountInfo;

        const showPercentage =
            hotelsPercentDiscount &&
            firstOfferDiscount?.percent &&
            firstOfferDiscount.percent >= MIN_PERCENT_TO_SHOW;

        return firstOfferPrice ? (
            <Flex inline alignItems="center" between={2} above={1}>
                <Price
                    className={cx('firstOfferPrice')}
                    {...firstOfferPrice}
                    color={
                        firstOfferDiscount && !showPercentage
                            ? 'alert'
                            : undefined
                    }
                    isFrom
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'firstOfferPrice',
                    })}
                />
                {firstOfferDiscount && (
                    <DiscountInfo
                        hotelsPercentDiscount={hotelsPercentDiscount}
                        discountInfo={firstOfferDiscount}
                        discountSize="s"
                        percentSize="s"
                    />
                )}
            </Flex>
        ) : (
            <span className={cx('firstOfferPrice', 'firstOfferPrice_noOffers')}>
                {isWithoutDatesPage
                    ? i18nShortViewBlock.findOutThePrice()
                    : i18nShortViewBlock.noOffersNew()}
            </span>
        );
    }, [minPrice, offers, isWithoutDatesPage]);

    return (
        <div
            className={cx('hotel', className)}
            ref={rootRef}
            style={style}
            {...prepareQaAttributes(props)}
        >
            {!hideImage && (
                <TravelImage
                    src={imageSrc}
                    className={cx('imageContainer', imageClassName)}
                    imageStub={imageStub}
                    imageAlt={name}
                    hasFitCover
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'photo',
                    })}
                />
            )}
            <div className={cx('hotelInfo')}>
                <div className={cx('hotelTitle')}>
                    <Link
                        className={cx('hotelName', titleClassName)}
                        url={hotelLandingUrl}
                        target="_blank"
                        onClick={handleClickHotelName}
                        rel="noopener noreferrer"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'hotelNameWithStars',
                        })}
                    >
                        {name}
                        {stars && (
                            <>
                                {CHAR_NBSP}
                                <HotelStars stars={stars} size="8" />
                            </>
                        )}
                    </Link>
                </div>
                <div className={cx('ratingsInfo')}>
                    {Boolean(rating) && (
                        <Rating
                            className={cx('rating')}
                            rating={rating}
                            size="s"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'rating',
                            })}
                        />
                    )}
                    {Boolean(totalTextReviewCount) && (
                        <span className={cx('reviewsCount')}>
                            {i18nBlock.totalReviewCount({
                                totalReviewCount: totalTextReviewCount,
                            })}
                            {CHAR_NBSP}
                            {CHAR_NBSP}
                            {CHAR_MIDDLE_DOT}
                            {CHAR_NBSP}
                            {CHAR_NBSP}
                        </span>
                    )}
                    {Boolean(categoryName) && (
                        <span
                            className={cx('categoryName')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'categoryName',
                            })}
                        >
                            {categoryName}
                        </span>
                    )}
                </div>
                {canRenderAddress && address && (
                    <div className={cx('address')}>{address}</div>
                )}
                {canRenderOfferPrice && priceBlock}
            </div>
        </div>
    );
};

export default HotelShortView;
