import React, {createRef, PureComponent, ReactNode} from 'react';

import {
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
    IChangeFilterGroupPayloadWithTarget,
    IChangePriceFilterPayloadWithTarget,
    IFullFilters,
} from 'types/hotels/search/IFiltersInfo';
import {IWithClassName} from 'types/withClassName';
import {TExperiments} from 'server/providers/experiments/types';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IResetFilterAction} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';

import {closeFilters} from 'i18n/hotels-SearchPageFilters';

import CloseIcon from 'icons/16/Close';
import Separator from 'components/Separator/Separator';
import HideBodyVerticalScroll from 'components/HideBodyVerticalScroll/HideBodyVerticalScroll';
import HotelsFiltersGroups from './components/HotelsFiltersGroups/HotelsFiltersGroups';
import DesktopHotelsFiltersContainer from './components/DesktopHotelsFiltersContainer/DesktopHotelsFiltersContainer';
import MobileHotelsFiltersContainer from './components/MobileHotelsFiltersContainer/MobileHotelsFiltersContainer';
import HotelsFiltersBar from './components/HotelsFiltersBar/HotelsFiltersBar';
import HotelsFiltersControlButtons from './components/HotelsFiltersControlButtons/HotelsFiltersControlButtons';
import HotelsFiltersBarSkeleton from './components/HotelsFiltersBarSkeleton/HotelsFiltersBarSkeleton';
import MobileHotelsFiltersBarSkeleton from './components/MobileHotelsFiltersBarSkeleton/MobileHotelsFiltersBarSkeleton';
import ToggleFiltersVisibilityButton from './components/ToggleFiltersVisibilityButton/ToggleFiltersVisibilityButton';
import HotelsQuickFilters from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/HotelsQuickFilters';
import HotelsPriceFilter from 'projects/hotels/components/HotelsFilters/components/PriceFilter/PriceFilter';
import HotelsPriceFilterRange, {
    IHotelsPriceFilterRangeProps,
} from 'projects/hotels/components/HotelsFilters/components/PriceFilter/components/Range';
import MobileHotelsFiltersScrollableSheet from './components/MobileHotelsFiltersScrollableSheet/MobileHotelsFiltersScrollableSheet';
import MobileHotelsFiltersCurtain from './components/MobileHotelsFiltersCurtain/MobileHotelsFiltersCurtain';

import cx from './HotelsFilters.scss';

interface IHotelsFiltersProps extends IWithClassName, IWithQaAttributes {
    nights: number;
    filters: IFullFilters;
    deviceInfo: IDevice;
    canDisableFilters: boolean;
    totalActiveFilters: number;
    experiments: TExperiments;
    mapListRadioButtonNode?: ReactNode;
    renderFiltersBar?: (props: {onToggleViewFilters: () => void}) => ReactNode;
    renderSort?: () => ReactNode;

    withNestedSheetsExp?: boolean;
    withScrollableSheetExp?: boolean;

    onChangeFilterGroup(
        payload: IChangeFilterGroupPayloadWithTarget,
        meta: IChangeFilterGroupMeta,
    ): void;
    onChangePriceFilter(payload: IChangePriceFilterPayloadWithTarget): void;
    onResetFilters(): void;
    onRevertFilters(): void;
    onApplyFilters(): void;

    onBaseCloseFilters?(): void;
    onToggleViewFilters?(): void;
    onChangeQuickFilter?(payload: IChangeFilterGroupPayload): void;
    onResetFilter(payload: IResetFilterAction): void;
}

interface IHotelsFiltersState {
    lockScroll: boolean;
    canRenderFiltersGroups: boolean;
    filtersContainerOffset: number;
}

class HotelsFilters extends PureComponent<
    IHotelsFiltersProps,
    IHotelsFiltersState
> {
    private canApplyFiltersAfterFinishAnimate: boolean = false;

    state: IHotelsFiltersState = {
        lockScroll: false,
        canRenderFiltersGroups: false,
        filtersContainerOffset: 70,
    };

    private rootRef = createRef<HTMLDivElement>();

    /* Handlers -> FiltersGroups */

    private handleApplyFiltersGroups = (): void => {
        const {
            deviceInfo: {isMobile},
        } = this.props;

        this.hideFiltersGroups();

        if (isMobile) {
            this.applyFilters();
        } else {
            this.canApplyFiltersAfterFinishAnimate = true;
        }
    };

    private handleFinishAnimateDesktopFiltersGroups = (): void => {
        if (this.canApplyFiltersAfterFinishAnimate) {
            this.applyFilters();

            this.canApplyFiltersAfterFinishAnimate = false;
        } else {
            this.revertFilters();
        }

        this.setState({
            lockScroll: false,
        });
    };

    private handleBaseCloseFilters = (): void => {
        const {onBaseCloseFilters} = this.props;

        onBaseCloseFilters?.();

        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_CROSS_CLICK);
    };

    private handleCloseMobileFilters = (): void => {
        this.hideFiltersGroups();
        this.revertFilters();
        this.handleBaseCloseFilters();
    };

    private handleToggleViewFilters = (): void => {
        this.setState(({canRenderFiltersGroups, lockScroll}) => ({
            canRenderFiltersGroups: !canRenderFiltersGroups,
            lockScroll: canRenderFiltersGroups ? lockScroll : !lockScroll,
        }));

        reachGoal(EHotelsGoal.SEARCH_PAGE_ALL_FILTERS_BUTTON_CLICK);
    };

    private handleCloseFilters = (): void => {
        this.setState({
            canRenderFiltersGroups: false,
        });

        this.handleBaseCloseFilters();
    };

    /* Handlers -> PriceFilter */

    private handleChangePriceFilter = ([minValue, maxValue]: [
        number,
        number,
    ]): void => {
        this.props.onChangePriceFilter({
            minValue,
            maxValue,
            targetFiltersType: 'CURRENT',
        });

        reachGoal(EHotelsGoal.SEARCH_PAGE_PRICE_FILTER_CHANGE);
    };

    /* Handlers -> FilterGroup */

    private handleChangeFilterGroup = (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ): void => {
        const {onChangeFilterGroup} = this.props;

        onChangeFilterGroup(
            {
                ...payload,
                targetFiltersType: 'CURRENT',
            },
            meta,
        );
    };

    /* Handlers -> QuickFilter */

    private handleChangeQuickFilter = (
        payload: IChangeFilterGroupPayload,
    ): void => {
        const {onChangeQuickFilter} = this.props;

        onChangeQuickFilter?.(payload);

        reachGoal(EHotelsGoal.SEARCH_PAGE_QUICK_FILTERS_CHANGE);
    };

    /* Actions */

    private revertFilters = (): void => {
        const {onRevertFilters} = this.props;

        onRevertFilters();
    };

    private handleResetFilters = (): void => {
        const {onResetFilters} = this.props;

        onResetFilters();
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_RESET);
    };

    private applyFilters = (): void => {
        const {onApplyFilters} = this.props;

        onApplyFilters();
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_APPLY);
    };

    private hideFiltersGroups = (): void => {
        this.setState({canRenderFiltersGroups: false});
    };

    /* Render FiltersBar */

    private renderFiltersBar(): React.ReactNode {
        const {
            deviceInfo,
            renderFiltersBar,
            mapListRadioButtonNode,
            filters: {detailedFiltersBatches},
        } = this.props;

        if (renderFiltersBar) {
            return renderFiltersBar({
                onToggleViewFilters: this.handleToggleViewFilters,
            });
        }

        if (!detailedFiltersBatches) {
            if (deviceInfo.isDesktop) {
                return (
                    <HotelsFiltersBarSkeleton
                        className={cx('filtersBarSkeleton')}
                    />
                );
            }

            return (
                <MobileHotelsFiltersBarSkeleton
                    className={cx('filtersBarSkeletonRedesign')}
                />
            );
        }

        return (
            <HotelsFiltersBar
                deviceInfo={deviceInfo}
                beforeFiltersNode={mapListRadioButtonNode}
                quickFiltersNode={this.renderQuickFiltersNode()}
                toggleFiltersVisibilityNode={this.renderToggleFiltersVisibilityButton()}
            />
        );
    }

    private renderPriceRangeFilter(): React.ReactNode {
        const {
            filters: {priceFilter},
            deviceInfo,
        } = this.props;

        if (!priceFilter) {
            return null;
        }

        const props: IHotelsPriceFilterRangeProps = {
            onChange: this.handleChangePriceFilter,
            currency: priceFilter.currency,
            value: [priceFilter.minValue ?? 0, priceFilter.maxValue ?? 0],
            min: priceFilter.minPriceEstimate,
            max: priceFilter.maxPriceEstimate,
        };

        return (
            <div className={cx(deviceMods('priceFilterContainer', deviceInfo))}>
                {deviceInfo.isDesktop ? (
                    <HotelsPriceFilterRange
                        className={cx('priceFilterRange')}
                        enableTooltip
                        {...props}
                    />
                ) : (
                    <HotelsPriceFilter
                        spaceBetween={5}
                        inputsSpaceBetween={4}
                        disableMobileRangeSpacing
                        {...props}
                    />
                )}
            </div>
        );
    }

    private renderQuickFiltersNode(): React.ReactNode {
        const {deviceInfo} = this.props;

        return (
            <HotelsQuickFilters
                // TODO Удалить после отказа от quickFilters
                dirtyPatchDesktopControls
                className={cx(
                    'quickFilters',
                    deviceMods('quickFilters', deviceInfo),
                )}
                onChange={this.handleChangeQuickFilter}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'quickFilters',
                })}
            />
        );
    }

    private renderToggleFiltersVisibilityButton(): React.ReactNode {
        const {canRenderFiltersGroups} = this.state;
        const {deviceInfo, totalActiveFilters} = this.props;

        return (
            <ToggleFiltersVisibilityButton
                deviceInfo={deviceInfo}
                totalActiveFilters={totalActiveFilters}
                canRenderFiltersGroups={canRenderFiltersGroups}
                onToggleViewFilters={this.handleToggleViewFilters}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'allFiltersButton',
                })}
            />
        );
    }

    private renderCloseFiltersButton(): React.ReactNode {
        return (
            <div className={cx('closeWrapper')}>
                <button
                    className={cx('closeButton')}
                    onClick={this.handleCloseFilters}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'closeFiltersButton',
                    })}
                >
                    {closeFilters()} <CloseIcon className={cx('closeIcon')} />
                </button>
            </div>
        );
    }

    /* Render DetailedFilters Containers */

    private renderMobileHotelsFiltersCurtain(): React.ReactNode {
        const {
            filters: {
                detailedFilters,
                activeFilterAtoms,
                foundHotelCount,
                quickFilters,
            },
            totalActiveFilters,
            onResetFilter,
            nights,
        } = this.props;

        if (!detailedFilters) {
            return null;
        }

        return (
            <MobileHotelsFiltersCurtain
                quickFilters={quickFilters}
                onQuickFilterChange={this.handleChangeQuickFilter}
                filterGroups={detailedFilters}
                activeFilterAtoms={activeFilterAtoms}
                foundHotelCount={foundHotelCount}
                onChangeFilterGroup={this.handleChangeFilterGroup}
                onResetAll={this.handleResetFilters}
                onResetFilter={onResetFilter}
                isResetDisabled={totalActiveFilters === 0}
                nights={nights}
                priceFilterNode={this.renderPriceRangeFilter()}
                isOpened={this.state.canRenderFiltersGroups}
                onClose={this.handleApplyFiltersGroups}
            />
        );
    }

    private renderMobileHotelsFiltersSheet(): React.ReactNode {
        const {
            filters: {
                detailedFilters,
                activeFilterAtoms,
                foundHotelCount,
                quickFilters,
            },
            totalActiveFilters,
        } = this.props;

        if (!detailedFilters) {
            return null;
        }

        return (
            <MobileHotelsFiltersScrollableSheet
                quickFilters={quickFilters}
                onQuickFilterChange={this.handleChangeQuickFilter}
                filtersGroupsNode={this.renderFiltersGroups()}
                activeFilterAtoms={activeFilterAtoms}
                foundHotelCount={foundHotelCount}
                onResetAll={this.handleResetFilters}
                isResetDisabled={totalActiveFilters === 0}
                isOpened={this.state.canRenderFiltersGroups}
                onClose={this.handleApplyFiltersGroups}
            />
        );
    }

    private renderMobileHotelsFiltersContainer(): React.ReactNode {
        const {withNestedSheetsExp, withScrollableSheetExp} = this.props;

        if (withNestedSheetsExp) {
            return this.renderMobileHotelsFiltersCurtain();
        }

        if (withScrollableSheetExp) {
            return this.renderMobileHotelsFiltersSheet();
        }

        const {canRenderFiltersGroups} = this.state;

        return (
            <MobileHotelsFiltersContainer
                isVisible={canRenderFiltersGroups}
                onClose={this.handleCloseMobileFilters}
            >
                <>
                    {this.renderFiltersGroups()}
                    {this.renderControlButtons()}
                </>
            </MobileHotelsFiltersContainer>
        );
    }

    private renderDesktopHotelsFiltersContainer(): React.ReactNode {
        const {canDisableFilters, deviceInfo} = this.props;
        const {lockScroll, canRenderFiltersGroups, filtersContainerOffset} =
            this.state;
        const filterGroups = this.renderFiltersGroups();

        if (!filterGroups) {
            return null;
        }

        return (
            <DesktopHotelsFiltersContainer
                className={cx('allFiltersContainer')}
                deviceType={deviceInfo}
                isVisible={canRenderFiltersGroups}
                topOffset={filtersContainerOffset}
                canDisableFilters={canDisableFilters}
                onClose={this.hideFiltersGroups}
                onFinishAnimate={this.handleFinishAnimateDesktopFiltersGroups}
            >
                {this.renderCloseFiltersButton()}
                {filterGroups}

                <div className={cx('controlButtonsWrapper')}>
                    <Separator className={cx('separator')} />
                    {this.renderControlButtons()}
                </div>

                {lockScroll && <HideBodyVerticalScroll />}
            </DesktopHotelsFiltersContainer>
        );
    }

    /* Render DetailedFilters */

    private renderFiltersGroups(): React.ReactNode {
        const {
            filters: {detailedFiltersBatches, activeFilterAtoms},
            deviceInfo,
            nights,
            withScrollableSheetExp,
        } = this.props;

        const {canRenderFiltersGroups} = this.state;

        if (detailedFiltersBatches) {
            return (
                <HotelsFiltersGroups
                    priceFilterNode={this.renderPriceRangeFilter()}
                    filterGroups={detailedFiltersBatches}
                    activeFilterAtoms={activeFilterAtoms}
                    isVisible={canRenderFiltersGroups}
                    deviceType={deviceInfo}
                    newDesign={deviceInfo.isMobile && withScrollableSheetExp}
                    nights={nights}
                    onChangeFilterGroup={this.handleChangeFilterGroup}
                    {...prepareQaAttributes(this.props)}
                />
            );
        }

        return null;
    }

    private renderControlButtons(): React.ReactNode {
        const {
            deviceInfo,
            totalActiveFilters,
            filters: {foundHotelCount},
        } = this.props;

        return (
            <HotelsFiltersControlButtons
                className={cx(
                    'controlButtons',
                    deviceMods('controlButtons', deviceInfo),
                )}
                onResetFilters={this.handleResetFilters}
                onApplyFilters={this.handleApplyFiltersGroups}
                deviceInfo={deviceInfo}
                foundHotelCount={foundHotelCount}
                isDisabledResetButton={totalActiveFilters === 0}
                {...prepareQaAttributes(this.props)}
            />
        );
    }

    render(): React.ReactNode {
        const {className, deviceInfo} = this.props;

        return (
            <div
                className={cx(
                    className,
                    'filters',
                    deviceMods('filters', deviceInfo),
                )}
                ref={this.rootRef}
            >
                {this.renderFiltersBar()}
                {deviceInfo.isMobile
                    ? this.renderMobileHotelsFiltersContainer()
                    : this.renderDesktopHotelsFiltersContainer()}
            </div>
        );
    }
}

export default HotelsFilters;
