import {Component, ChangeEvent, ReactNode} from 'react';
import _find from 'lodash/find';

import {IWithClassName} from 'types/withClassName';
import {TAnyArrayElement} from 'types/TArrayElement';

import {RadioboxProps} from '@yandex-lego/components/Radiobox';
import Radiobox from 'components/Radiobox/Radiobox';

/* Init styles */
import cx from './BaseRadioBox.scss';

type TWithId<OptionId> = {id: Extract<OptionId, string>};
type TOption<OptionType, OptionId> = OptionType & TWithId<OptionId>;

/* Components */
interface IBaseRadioBoxProps<OptionType, OptionId> extends IWithClassName {
    id?: string;
    radioClassName?: string;
    options: TOption<OptionType, OptionId>[];
    renderRadioLabel: (option: OptionType) => ReactNode;
    getRadioDisabled: (option: OptionType) => boolean;
    value?: Extract<OptionId, string>;
    onChange: (payload: {option: OptionType}) => void;
}

class BaseRadioBox<OptionType, OptionId> extends Component<
    IBaseRadioBoxProps<OptionType, OptionId>
> {
    /* Handlers */
    private handleChange = (e: ChangeEvent<HTMLInputElement>): void => {
        const {
            target: {value: optionId},
        } = e;
        const {onChange, options} = this.props;
        const option = _find<TOption<OptionType, OptionId>>(
            options,
            ({id}: TWithId<OptionId>) => String(id) === String(optionId),
        );

        if (option) {
            onChange({option});
        }
    };

    private mapOption = (
        option: TOption<OptionType, OptionId>,
    ): TAnyArrayElement<RadioboxProps['options']> => {
        const {getRadioDisabled, renderRadioLabel, radioClassName} = this.props;
        const disabled = getRadioDisabled(option);
        const {id} = option;

        return {
            value: id,
            label: renderRadioLabel(option),
            className: cx(radioClassName, 'radio'),
            disabled,
        };
    };

    render(): ReactNode {
        const {options, className, value} = this.props;

        return (
            <Radiobox
                className={className}
                value={value}
                onChange={this.handleChange}
                options={options.map(this.mapOption)}
            />
        );
    }
}

export default BaseRadioBox;
