import {PureComponent, ReactNode} from 'react';

/* Types */
import {
    TActiveFilterAtomsType,
    IBasicFilter,
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
} from 'types/hotels/search/IFiltersInfo';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

/* Components */
import BaseCheckboxList from '../BaseCheckboxList/BaseCheckboxList';

/* Component Types */
interface IHotelsCheckboxListProps extends IWithDeviceType, IWithQaAttributes {
    id: string;
    items: IBasicFilter[];
    activeFilterAtoms: TActiveFilterAtomsType;
    onChange: (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ) => void;
    renderCheckboxLabel?: (option: IBasicFilter) => ReactNode;
}

/* Init styles */
import cx from './HotelsCheckboxList.scss';

class HotelsCheckboxList extends PureComponent<IHotelsCheckboxListProps> {
    private handleChange = ({
        item,
        checked,
    }: {
        item: IBasicFilter;
        checked: boolean;
    }): void => {
        const {onChange, id} = this.props;
        const {atoms} = item;

        onChange(checked ? {atomsOn: atoms} : {atomsOff: atoms}, {
            filterId: id,
            itemId: item.id,
        });
    };

    getCheckboxChecked = (item: IBasicFilter): boolean => {
        const {activeFilterAtoms} = this.props;
        const {atoms} = item;

        return atoms.every(atom => activeFilterAtoms[atom]);
    };

    getCheckboxDisabled = (item: IBasicFilter): boolean => !item.enabled;

    renderCheckboxLabel = (item: IBasicFilter): ReactNode => {
        const {renderCheckboxLabel} = this.props;
        const {id, name, hint} = item;

        return (
            <div
                className={cx('checkboxName')}
                {...prepareQaAttributes({parent: this.props, current: id})}
            >
                {renderCheckboxLabel ? renderCheckboxLabel(item) : name}
                {Boolean(hint) && (
                    <span className={cx('checkboxHint')}>{hint}</span>
                )}
            </div>
        );
    };

    render(): ReactNode {
        const {items, deviceType} = this.props;

        return (
            <BaseCheckboxList<IBasicFilter, string>
                className={cx('group', deviceMods('group', deviceType))}
                checkboxClassName={cx('checkbox')}
                items={items}
                onChange={this.handleChange}
                getCheckboxDisabled={this.getCheckboxDisabled}
                getCheckboxChecked={this.getCheckboxChecked}
                renderCheckboxLabel={this.renderCheckboxLabel}
            />
        );
    }
}

export default HotelsCheckboxList;
