import {PureComponent, ReactNode} from 'react';
import {flatten} from 'lodash';

import {
    EFilterTypes,
    EStarFilterIds,
    getStarsByFilterId,
} from 'projects/hotels/constants/filters';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {
    TActiveFilterAtomsType,
    EDetailedFiltersBatchItemType,
    IBasicFilter,
    IBasicFilterGroup,
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
    IDetailedFiltersBatch,
    IDetailedFilterItem,
} from 'types/hotels/search/IFiltersInfo';
import {IWithDeviceType} from 'types/withDeviceType';

import scrollTo from 'utilities/dom/scrollTo';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import HotelStars from 'components/HotelStars/HotelStars';
import HotelsPriceFilterGroup from 'projects/hotels/components/HotelsFilters/components/HotelsFiltersGroups/HotelsPriceFilterGroup';
import HotelsFiltersGroup from 'projects/hotels/components/HotelsFilters/components/HotelsFiltersGroup/HotelsFiltersGroup';

import RatingFilterLabel from '../RatingFilterLabel/RatingFilterLabel';

import cx from './HotelsFiltersGroups.scss';

/* Component Types */
interface IHotelsFiltersGroupsProps extends IWithDeviceType, IWithQaAttributes {
    className?: string;
    filterGroups: IDetailedFiltersBatch[];
    onChangeFilterGroup: (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ) => void;
    nights: number;
    activeFilterAtoms: TActiveFilterAtomsType;
    priceFilterNode: ReactNode;
    isVisible: boolean;
    newDesign?: boolean;
}

class HotelsFiltersGroups extends PureComponent<IHotelsFiltersGroupsProps> {
    private scrollContainerNode: HTMLDivElement | null = null;

    componentDidMount(): void {
        if (this.scrollContainerNode) {
            this.scrollContainerNode.addEventListener(
                'scroll',
                this.handleScroll,
            );
        }
    }

    componentDidUpdate(prevProps: Readonly<IHotelsFiltersGroupsProps>): void {
        if (
            !prevProps.isVisible &&
            this.props.isVisible &&
            this.scrollContainerNode
        ) {
            scrollTo({top: 0}, this.scrollContainerNode);
        }
    }

    componentWillUnmount(): void {
        this.unsubscribe();
    }

    setScrollContainerNodeRef = (dayNode: HTMLDivElement | null): void => {
        this.scrollContainerNode = dayNode;
    };

    private unsubscribe(): void {
        if (this.scrollContainerNode) {
            this.scrollContainerNode.removeEventListener(
                'scroll',
                this.handleScroll,
            );
        }
    }

    private handleScroll = (): void => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTER_GROUP_SCROLL);
        this.unsubscribe();
    };

    private renderFiltersGroup = (
        filterGroup: IBasicFilterGroup,
    ): ReactNode => {
        const {onChangeFilterGroup, activeFilterAtoms} = this.props;
        const {id} = filterGroup;

        return (
            <HotelsFiltersGroup
                key={filterGroup.id}
                filterGroup={filterGroup}
                activeFilterAtoms={activeFilterAtoms}
                onChange={onChangeFilterGroup}
                renderCheckboxLabel={
                    id === EFilterTypes.STARS
                        ? this.renderStarsFilterItem
                        : undefined
                }
                renderRadioLabel={
                    id === EFilterTypes.RATINGS
                        ? this.renderRatingFilterItem
                        : undefined
                }
                {...prepareQaAttributes(this.props)}
            />
        );
    };

    private renderRatingFilterItem = (item: IBasicFilter): ReactNode => {
        return (
            <RatingFilterLabel
                className={cx('ratingFilterLabel')}
                item={item}
            />
        );
    };

    private renderStarsFilterItem = ({id, name}: IBasicFilter): ReactNode => {
        const stars = getStarsByFilterId(id as EStarFilterIds);

        if (!stars) {
            return name;
        }

        return <HotelStars size="8" stars={stars} />;
    };

    private renderFiltersGroups(): ReactNode {
        const {filterGroups, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'filtersGroups',
                    `filtersGroups_columns_${filterGroups.length}`,
                )}
            >
                {deviceType.isMobile && (
                    <>
                        {this.isWithoutPriceFilter() &&
                            this.renderFiltersColumn(
                                [
                                    {
                                        type: EDetailedFiltersBatchItemType.PRICE,
                                    },
                                ],
                                -1,
                            )}
                    </>
                )}

                {filterGroups.map((group, index) =>
                    this.renderFiltersColumn(group.items, index),
                )}
            </div>
        );
    }

    private renderFiltersColumn(
        filters: IDetailedFilterItem[],
        index: number,
    ): ReactNode {
        return (
            <div className={cx('column')} key={index}>
                {filters.map(({type, detailedFilters: filter}) => {
                    if (type === EDetailedFiltersBatchItemType.PRICE) {
                        return this.renderPriceFilter();
                    }

                    if (!filter) {
                        return null;
                    }

                    const filtersGroupNode = this.renderFiltersGroup(filter);

                    if (filtersGroupNode) {
                        return (
                            <div
                                className={cx('group')}
                                key={filter.id}
                                {...prepareQaAttributes({
                                    parent: this.props,
                                    current: filter.id,
                                })}
                            >
                                <div className={cx('groupTitle')}>
                                    {filter.name}
                                </div>
                                {filtersGroupNode}
                            </div>
                        );
                    }

                    return null;
                })}
            </div>
        );
    }

    private renderPriceFilter(): ReactNode {
        const {deviceType, nights, priceFilterNode, newDesign} = this.props;

        return (
            <HotelsPriceFilterGroup
                key="price"
                nights={nights}
                newDesign={newDesign}
                deviceType={deviceType}
            >
                {priceFilterNode}
            </HotelsPriceFilterGroup>
        );
    }

    private isWithoutPriceFilter(): boolean {
        const {filterGroups} = this.props;

        return !flatten(filterGroups.map(item => item.items)).find(
            item => item.type === EDetailedFiltersBatchItemType.PRICE,
        );
    }

    render(): ReactNode {
        const {deviceType} = this.props;

        return (
            <div
                className={cx(
                    'filtersGroupsContent',
                    deviceMods('filtersGroupsContent', deviceType),
                )}
                ref={this.setScrollContainerNodeRef}
            >
                {this.renderFiltersGroups()}
            </div>
        );
    }
}

export default HotelsFiltersGroups;
