import {FC, memo, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {
    EQuickControlType,
    IChangeFilterGroupPayload,
    TQuickControlTypes,
    TQuickFilter,
} from 'types/hotels/search/IFiltersInfo';

import {
    activeFilterAtomsSelector,
    quickFiltersControlsSelector,
} from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/selectors';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getPatchedQuickControls from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/utilities/getPatchedQuickControls';

import QuickFilterPriceControl from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/components/Price/Price';
import QuickFilterItem, {
    IQuickFilterItemProps,
} from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/components/Item/Item';
import Flex from 'components/Flex/Flex';
import HotelsSort from 'projects/hotels/components/HotelsSort/HotelsSort';
import {TButtonTheme} from 'components/Button/Button';

export interface IHotelsQuickFiltersProps
    extends IWithClassName,
        IWithQaAttributes {
    theme?: TButtonTheme;
    onChange?(payload: IChangeFilterGroupPayload): void;

    dirtyPatchDesktopControls?: boolean;

    withSort?: boolean;
    withPrice?: boolean;
    size?: IQuickFilterItemProps['buttonSize'];
}

const HotelsQuickFilters: FC<IHotelsQuickFiltersProps> = memo(
    ({
        dirtyPatchDesktopControls,
        onChange,
        className,
        theme,
        withSort = true,
        withPrice = true,
        ...props
    }) => {
        const filters = useSelector(quickFiltersControlsSelector);
        const activeAtoms = useSelector(activeFilterAtomsSelector);
        // TODO Удалить после стабилизации апи и удаления quickFilters
        const patchedControls = useMemo(
            () =>
                dirtyPatchDesktopControls
                    ? getPatchedQuickControls(
                          filters.controls,
                          filters.oldFilters,
                      )
                    : filters.controls,
            [filters, dirtyPatchDesktopControls],
        );

        const isChecked = ({atomsOn, atomsOff}: TQuickFilter): boolean =>
            atomsOn.every(atom => activeAtoms[atom]) &&
            atomsOff.every(atom => !activeAtoms[atom]);

        const renderItem = (item: TQuickControlTypes): JSX.Element | false => {
            switch (item.type) {
                case EQuickControlType.QUICK_SORT:
                    return (
                        withSort && (
                            <HotelsSort
                                theme={theme}
                                key={item.id}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: item.id,
                                })}
                            />
                        )
                    );
                case EQuickControlType.QUICK_FILTER_PRICE:
                    return (
                        withPrice && (
                            <QuickFilterPriceControl
                                theme={theme}
                                key={item.id}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: item.id,
                                })}
                            />
                        )
                    );
                default:
                    return (
                        <QuickFilterItem
                            theme={theme}
                            key={item.id}
                            item={item}
                            onChange={onChange}
                            checked={isChecked(item)}
                            {...prepareQaAttributes({
                                parent: props,
                                current: item.id,
                            })}
                        />
                    );
            }
        };

        return (
            <Flex
                className={className}
                inline
                between={2}
                flexWrap="nowrap"
                {...prepareQaAttributes(props)}
            >
                {patchedControls.map(renderItem)}
            </Flex>
        );
    },
);

HotelsQuickFilters.displayName = 'HotelsQuickFilters';

export default HotelsQuickFilters;
