import {FC, memo, useRef} from 'react';

import {
    IChangeFilterGroupPayload,
    TQuickFilter,
} from 'types/hotels/search/IFiltersInfo';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useWindowLayout} from 'utilities/hooks/useWindowLayout';
import getItemButtonIconProps from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/utilities/getItemButtonIconProps';
import getItemHint from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/utilities/getItemHint';
import {deviceModMobile} from 'utilities/stylesUtils';
import getItemText from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/utilities/getItemText';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';
import {TButtonTheme} from 'components/Button/Button';
import QuickFilterItemButton, {
    IQuickFilterItemButtonProps,
} from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/components/Item/Button';

import cx from '../../HotelsQuickFilters.scss';

export interface IQuickFilterItemProps extends IWithQaAttributes {
    item: TQuickFilter;
    checked: boolean;
    className?: string;
    theme?: TButtonTheme;
    onChange?(value: IChangeFilterGroupPayload): void;
    buttonSize?: IQuickFilterItemButtonProps['size'];
}

const QuickFilterItem: FC<IQuickFilterItemProps> = ({
    item,
    item: {atomsOn, atomsOff, enabled, effect},
    checked,
    onChange,
    theme,
    className,
    buttonSize,
    ...props
}) => {
    const buttonRef = useRef<HTMLDivElement>(null);
    const hintVisible = useBoolean(false);
    const deviceType = useDeviceType();
    const {width} = useWindowLayout();

    const {isDesktop, isMobile} = deviceType;
    const iconProps = getItemButtonIconProps(item, isDesktop);
    const hint = getItemHint(item, isDesktop);
    const text = getItemText(item, width);

    const handleClick = useImmutableCallback(() => {
        onChange?.(checked ? {atomsOff: atomsOn} : {atomsOn, atomsOff});
    });

    return (
        <>
            <div
                ref={buttonRef}
                onMouseEnter={isDesktop ? hintVisible.setTrue : undefined}
                onMouseLeave={isDesktop ? hintVisible.setFalse : undefined}
                onClick={isMobile ? hintVisible.toggle : undefined}
                {...prepareQaAttributes({...props})}
            >
                <QuickFilterItemButton
                    name={effect}
                    className={className}
                    checked={checked}
                    disabled={!enabled}
                    onClick={handleClick}
                    size={buttonSize}
                    theme={theme}
                    {...iconProps}
                >
                    {text}
                </QuickFilterItemButton>
            </div>

            {!isDesktop && !enabled && hint && (
                <MessageBoxPopupOrBottomSheet
                    isVisible={hintVisible.value}
                    onClose={hintVisible.setFalse}
                    messageBoxProps={{
                        anchorRef: buttonRef,
                        direction: EPopupDirection.BOTTOM,
                    }}
                >
                    <div
                        className={cx(
                            'toggle_content',
                            deviceModMobile('toggle_content', deviceType),
                        )}
                    >
                        {hint}
                    </div>
                </MessageBoxPopupOrBottomSheet>
            )}
        </>
    );
};

export default memo(QuickFilterItem);
