import React, {RefObject, useCallback} from 'react';

import {
    EBasicFilterGroupId,
    IBasicFilter,
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
    IDetailedFilterItem,
    TActiveFilterAtomsType,
} from 'types/hotels/search/IFiltersInfo';
import {
    EResetFilterActionEffect,
    IResetFilterAction,
} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {calculateTotalFilterGroups} from 'projects/hotels/utilities/filters/calculateTotalFilterGroups';
import getActiveAtomsForGroup from 'projects/hotels/utilities/filters/getActiveAtomsForGroup';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import * as i18nBlock from 'i18n/hotels-SearchPageFilters';

import FilterNestedBottomSheet from 'components/FiltersBottomSheet/components/FilterNestedBottomSheet/FilterNestedBottomSheet';
import HotelsFiltersGroup from 'projects/hotels/components/HotelsFilters/components/HotelsFiltersGroup/HotelsFiltersGroup';
import RatingFilterLabel from 'projects/hotels/components/HotelsFilters/components/RatingFilterLabel/RatingFilterLabel';

interface IMobileHotelsBottomSheetFilter {
    filterGroup: IDetailedFilterItem;
    bottomSheetRef: RefObject<HTMLDivElement>;
    foundHotelCount: number;
    activeFilterAtoms: TActiveFilterAtomsType;
    onChange: (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ) => void;
    onReset: (payload: IResetFilterAction) => void;
}

const MobileHotelsBottomSheetFilter: React.FC<IMobileHotelsBottomSheetFilter> =
    ({
        filterGroup,
        bottomSheetRef,
        foundHotelCount,
        activeFilterAtoms,
        onChange,
        onReset,
    }) => {
        const handleReset = useCallback(() => {
            if (!filterGroup.detailedFilters) {
                return;
            }

            const action: IResetFilterAction = {
                name: filterGroup.detailedFilters.name,
                effect: EResetFilterActionEffect.SIMPLE,
                atomsOff: getActiveAtomsForGroup(
                    filterGroup.detailedFilters,
                    activeFilterAtoms,
                ),
                needResetFilterPrice: true,
                needResetGeoIdFilter: true,
            };

            onReset(action);
        }, [onReset, filterGroup, activeFilterAtoms]);

        const renderRadioLabel = useImmutableCallback(
            (option: IBasicFilter): React.ReactNode => {
                return <RatingFilterLabel item={option} />;
            },
        );

        if (!filterGroup.detailedFilters) {
            return null;
        }

        const isRating =
            filterGroup.detailedFilters.id === EBasicFilterGroupId.RATINGS;

        const selectedCount = calculateTotalFilterGroups(
            [filterGroup.detailedFilters],
            activeFilterAtoms,
        );

        return (
            <FilterNestedBottomSheet
                label={filterGroup.detailedFilters?.name}
                selectedCount={selectedCount}
                onReset={handleReset}
                firstBottomSheetRef={bottomSheetRef}
                submitButtonDisabled={!foundHotelCount}
                submitButtonLabel={i18nBlock.done()}
                withArrow
            >
                <HotelsFiltersGroup
                    key={filterGroup.detailedFilters.name}
                    filterGroup={filterGroup.detailedFilters}
                    activeFilterAtoms={activeFilterAtoms}
                    onChange={onChange}
                    renderRadioLabel={isRating ? renderRadioLabel : undefined}
                />
            </FilterNestedBottomSheet>
        );
    };

export default MobileHotelsBottomSheetFilter;
