import React, {useMemo, useRef} from 'react';

import {
    EDetailedFiltersBatchItemType,
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
    IDetailedFilterItem,
    TQuickFilter,
    TActiveFilterAtomsType,
} from 'types/hotels/search/IFiltersInfo';
import {IResetFilterAction} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/common-filters';
import * as i18nBlockHotel from 'i18n/hotels-SearchPageFilters';

import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import FiltersBottomSheet from 'components/FiltersBottomSheet/FiltersBottomSheet';
import QuickFiltersPadder from 'components/FiltersBottomSheet/components/QuickFiltersPadder/QuickFiltersPadder';
import MobileHotelsBottomSheetFilter from 'projects/hotels/components/HotelsFilters/components/MobileHotelsBottomSheetFilter/MobileHotelsBottomSheetFilter';
import HotelsPriceFilterGroup from 'projects/hotels/components/HotelsFilters/components/HotelsFiltersGroups/HotelsPriceFilterGroup';
import MobileHotelsQuickCurtainFilters from 'projects/hotels/components/HotelsFilters/components/MobileHotelsQuickCurtainFilters/MobileHotelsQuickCurtainFilters';

import cx from './MobileHotelsFiltersCurtain.scss';

interface IProps {
    filterGroups: IDetailedFilterItem[];
    quickFilters?: TQuickFilter[];
    foundHotelCount?: number;

    onChangeFilterGroup: (
        payload: IChangeFilterGroupPayload,
        meta: IChangeFilterGroupMeta,
    ) => void;
    onQuickFilterChange(payload: IChangeFilterGroupPayload): void;
    activeFilterAtoms: TActiveFilterAtomsType;
    onResetAll: () => void;
    onClose: () => void;
    onResetFilter: (payload: IResetFilterAction) => void;
    isResetDisabled?: boolean;
    priceFilterNode?: React.ReactNode;
    nights: number;
    isOpened: boolean;
}

const MobileHotelsFiltersCurtain: React.FC<IProps> = ({
    filterGroups,
    activeFilterAtoms,
    onChangeFilterGroup,
    foundHotelCount,
    onResetAll,
    onResetFilter,
    isResetDisabled,
    nights,
    priceFilterNode,
    isOpened,
    onClose,
    quickFilters,
    onQuickFilterChange,
}) => {
    const deviceType = useDeviceType();

    const priceFilter = useMemo(
        () =>
            filterGroups.find(
                ({type}) => type === EDetailedFiltersBatchItemType.PRICE,
            ),
        [filterGroups],
    );

    const groups = useMemo(
        () =>
            filterGroups.filter(
                ({type}) => type === EDetailedFiltersBatchItemType.GROUP,
            ),
        [filterGroups],
    );

    const bottomSheetRef = useRef<HTMLDivElement>(null);

    const submitLabel = foundHotelCount
        ? i18nBlockHotel.showVariants({count: foundHotelCount})
        : i18nBlockHotel.noVariants();

    return (
        <FiltersBottomSheet
            isOpened={isOpened}
            onClose={onClose}
            header={i18nBlock.title()}
            onReset={onResetAll}
            resetButtonDisabled={Boolean(isResetDisabled)}
            submitLabel={submitLabel}
            onSubmit={onClose}
            submitButtonDisabled={!foundHotelCount}
            outerSheetRef={bottomSheetRef}
            contentPadding="none"
        >
            {quickFilters && (
                <QuickFiltersPadder>
                    <MobileHotelsQuickCurtainFilters
                        onChange={onQuickFilterChange}
                    />
                </QuickFiltersPadder>
            )}

            <div className={cx('content')}>
                {priceFilter && priceFilterNode && (
                    <HotelsPriceFilterGroup
                        newDesign
                        deviceType={deviceType}
                        nights={nights}
                    >
                        {priceFilterNode}
                    </HotelsPriceFilterGroup>
                )}
                <Intersperse separator={<Separator />}>
                    {groups.map(filter => (
                        <MobileHotelsBottomSheetFilter
                            key={filter.detailedFilters?.name}
                            filterGroup={filter}
                            bottomSheetRef={bottomSheetRef}
                            onChange={onChangeFilterGroup}
                            activeFilterAtoms={activeFilterAtoms}
                            foundHotelCount={foundHotelCount || 0}
                            onReset={onResetFilter}
                        />
                    ))}
                </Intersperse>
            </div>
        </FiltersBottomSheet>
    );
};

export default MobileHotelsFiltersCurtain;
