import {FC, memo, useCallback, useRef, useState} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';

import Flex from 'components/Flex/Flex';
import HotelsPriceFilterRangeInputs from 'projects/hotels/components/HotelsFilters/components/PriceFilter/components/RangeInputs';
import HotelsPriceFilterRange from 'projects/hotels/components/HotelsFilters/components/PriceFilter/components/Range';
import {TBoxSizes} from 'components/Box/Box';

import cx from './PriceFilter.scss';

export interface IHotelsPriceFilterProps
    extends IWithQaAttributes,
        IWithClassName {
    max: number;
    min: number;
    value: [number, number];
    currency: CurrencyType;
    disabled?: boolean;
    onChange(value: [number, number]): void;

    // TODO Узнать - почему в одном месте (модалка цены) есть отступы, а в остальных нет?
    disableMobileRangeSpacing?: boolean;
    // TODO Удалить после синхронизации отступов
    inputsSpaceBetween?: TBoxSizes;
    // TODO Удалить после синхронизации отступов
    spaceBetween?: TBoxSizes;
}

const HotelsPriceFilter: FC<IHotelsPriceFilterProps> = ({
    min,
    max,
    value,
    disabled,
    currency,
    onChange,
    className,
    spaceBetween,
    inputsSpaceBetween,
    disableMobileRangeSpacing,
}: IHotelsPriceFilterProps) => {
    const [updatedRangeValue, setUpdatedRangeValue] = useState<
        [number, number] | null
    >(null);
    const [updatedInputsValue, setUpdatedInputsValue] = useState<
        [number, number] | null
    >(null);
    const handleChange = useImmutableCallback((nextValue: [number, number]) => {
        setUpdatedInputsValue(null);
        setUpdatedRangeValue(null);
        onChange(nextValue);
    });
    const deviceType = useDeviceType();
    /**
     * TODO Удалить, ещё один костыль из-за слайдера
     * Как должно быть
     * - Юзер жмёт на ползунок, у него есть tabIndex -> фокус автоматически переходит на ползунок
     * - Кстати, из-за этого можно почти бесшовно сделать поддержку стрелочек вправо-влево
     * Как сейчас
     * - Юзер жмёт на ползунок, а слайдер обрывает это событие -> фокус остаётся в инпуте
     * - Приветствуем, о, новый костыль, при апдейте (при апдейте, Карл) слайдера мы проверяем, а не в фокусе ли инпут,
     * если в фокусе - сбрасываем фокус
     */
    const inputsRef = useRef<HTMLDivElement | null>(null);

    const tryBlurInputs = useCallback(() => {
        const container = inputsRef.current;
        const activeElement = document.activeElement;

        if (activeElement && container?.contains(activeElement)) {
            (activeElement as HTMLElement).blur();
        }
    }, []);

    const handleRangeUpdate = useImmutableCallback(
        (nextValue: [number, number]) => {
            tryBlurInputs();
            setUpdatedRangeValue(nextValue);
        },
    );

    return (
        <Flex
            flexDirection="column"
            between={spaceBetween ?? (deviceType.isMobile ? 9 : 5)}
            className={className}
        >
            <HotelsPriceFilterRangeInputs
                max={max}
                min={min}
                ref={inputsRef}
                value={updatedRangeValue ?? value}
                onChange={handleChange}
                disabled={disabled}
                currency={currency}
                spaceBetween={inputsSpaceBetween}
            />
            <HotelsPriceFilterRange
                max={max}
                min={min}
                value={updatedInputsValue ?? value}
                currency={currency}
                onUpdate={handleRangeUpdate}
                onChange={handleChange}
                disabled={disabled}
                className={cx(
                    'range',
                    disableMobileRangeSpacing && 'disable_mobile_spacing',
                    deviceModMobile('range', deviceType),
                )}
            />
        </Flex>
    );
};

export default memo(HotelsPriceFilter);
